<?php
namespace Newland\Toubiz\Poi\Neos\Controller;

/*
 * This file is part of the "toubiz-poi-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\ContentRepository\Domain\Model\Node;
use Neos\Flow\Annotations as Flow;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Sync\Neos\Domain\Filter\ArticleFilter;
use Newland\Toubiz\Sync\Neos\Domain\Filter\CategoryFilter;
use Newland\Toubiz\Sync\Neos\Domain\Model\Article;
use Newland\Toubiz\Sync\Neos\Domain\Repository\ArticleRepository;
use Newland\Toubiz\Sync\Neos\Domain\Repository\CategoryRepository;

/**
 * Point of interests controller.
 *
 * @Flow\Scope("singleton")
 */
class PointOfInterestsController extends AbstractActionController
{
    /**
     * @var int Number of results per page.
     */
    const PAGE_ITEMS = 10;

    /**
     * @Flow\Inject
     * @var ArticleRepository
     */
    protected $articleRepository;

    /**
     * @Flow\Inject
     * @var CategoryRepository
     */
    protected $categoryRepository;

    /**
     * Teaser action.
     *
     * Renders a single teaser.
     *
     * @return void
     */
    public function teaserAction()
    {
        if (array_key_exists('article', $this->properties)) {
            $this->view->assign('article', $this->articleRepository->find($this->properties['article']));
        }
        $this->view->assignMultiple([
            'wrapStart' => $this->request->getInternalArgument('__wrapStart'),
            'wrapEnd' => $this->request->getInternalArgument('__wrapEnd')
        ]);
    }

    /**
     * Teaser list action.
     *
     * Renders a list of teasers.
     *
     * @return void
     */
    public function teaserListAction()
    {
        $filter = new ArticleFilter;
        if (array_key_exists('categoryFilter', $this->properties)) {
            $filter->setCategories($this->properties['categoryFilter']);
        }
        if (array_key_exists('recordLimit', $this->properties)) {
            $filter->setLimit($this->properties['recordLimit']);
        }

        $articles = $this->articleRepository->findByFilter($filter);
        $this->view->assign('articles', $articles);
    }

    /**
     * List action.
     *
     * Lists articles and allows filtering.
     *
     * @return void
     */
    public function listAction()
    {
        $articleFilter = new ArticleFilter;
        $articleFilter->setLimit(self::PAGE_ITEMS);

        if (array_key_exists('articleType', $this->properties)) {
            $articleFilter->setMainType((int)$this->properties['articleType']);
        }

        $this->view->assignMultiple(
            [
                'articles' => $this->articleRepository->findByFilter($articleFilter),
                'pagination' => [
                    'isFirst' => true,
                    'page' => 1,
                    'isLast' => false,
                    'count' => $this->articleRepository->countByFilter($articleFilter, self::PAGE_ITEMS)
                ]
            ]);

        $categoryFilter = new CategoryFilter;
        if (array_key_exists('articleType', $this->properties)) {
            $categoryFilter->setMainType($this->properties['articleType']);
        }
        $categoryFilter->setOrderBy(['category.title' => 'asc']);
        $categories = $this->categoryRepository->findByFilter($categoryFilter);
        $this->view->assign('categories', $categories);
    }

    /**
     * Filter list action.
     *
     * Returns articles based on given filters.
     *
     * @param Node $node
     * @param array $query  The filter query.
     * @return string
     */
    public function filterListAction(Node $node, array $query = [])
    {
        $this->initializeAction($node);

        $articleFilter = new ArticleFilter;
        $articleFilter->setLimit(self::PAGE_ITEMS);

        if (array_key_exists('articleType', $this->properties)) {
            $articleFilter->setMainType($this->properties['articleType']);
        }
        if (array_key_exists('categories', $query) && is_array($query['categories'])) {
            $articleFilter->setCategories($query['categories']);
        }
        if (array_key_exists('page', $query)) {
            $page = (int)$query['page'];
            $articleFilter->setOffset(($page - 1) * self::PAGE_ITEMS);
        } else {
            $page = 1;
        }

        $articles = $this->articleRepository->findByFilter($articleFilter);
        $articleCounts = $this->articleRepository->countByFilter($articleFilter, self::PAGE_ITEMS);

        $view = $this->createStandaloneView('PointOfInterests/FilterList');
        $view->assignMultiple(
            [
                'articles' => $articles,
                'pagination' => [
                    'isFirst' => ($page == 1),
                    'page' => $page,
                    'isLast' => $articleCounts['pages'] <= $page,
                    'count' => $articleCounts
                ]
            ]
        );
        return $view->render();
    }

    /**
     * Show action.
     *
     * Displays an article.
     *
     * @param Article $article
     * @return void
     */
    public function showAction(Article $article = null)
    {
        $this->view->assign('article', $article);

        if ($article->hasLatitude() && $article->hasLongitude()) {
            $onSiteFilter = new ArticleFilter;
            $onSiteFilter->setLatitude($article->getLatitude())
                         ->setLongitude($article->getLongitude())
                         ->setCoordinatePrecision(4)
                         ->setExcludes([$article]);

            $this->view->assign(
                'onSiteArticles',
                $this->articleRepository->findByFilter($onSiteFilter)
            );
        }

        if ($article->getMainType() == ArticleConstants::TYPE_TOUR) {
            $this->assignSimilarTours($article);
        }
    }

    /**
     * Assigns similar tours to the given one to the view.
     *
     * @param Article $tour
     * @return void
     */
    protected function assignSimilarTours(Article $tour)
    {
        $filter = new ArticleFilter;
        $filter->setCategoryCollection($tour->getCategories())
               ->setExcludes([$tour])
               ->setLimit(4);

        $this->view->assign(
            'similarTours',
            $this->articleRepository->findByFilter($filter)
        );
    }
}
