<?php declare(strict_types=1);

namespace Newland\Toubiz\Packages\Neos\LinkHandler;

use Neos\ContentRepository\Domain\Model\NodeInterface;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Mvc\Controller\ControllerContext;
use Newland\NeosCommon\LinkHandler\Domain\Model\Record;
use Newland\NeosCommon\LinkHandler\Handler\AbstractLinkHandler;
use Newland\Toubiz\Packages\Neos\Service\PackageUrlService;
use Newland\Toubiz\Sync\Neos\Domain\Model\Package;
use Newland\Toubiz\Sync\Neos\Domain\Repository\PackageRepository;

/**
 * @Flow\Scope("singleton")
 */
class PackageLinkHandler extends AbstractLinkHandler
{

    /**
     * @var PackageRepository
     * @Flow\Inject()
     */
    protected $packageRepository;

    /**
     * @var PackageUrlService
     * @Flow\Inject()
     */
    protected $packageUrlService;

    public function findByIdentifier(string $recordIdentifier): ?Record
    {
        $package = $this->packageRepository->findByIdentifier($recordIdentifier);
        if (!$package) {
            return null;
        }

        return new PackageRecord($package);
    }

    public function findBySearchTerm(string $searchTerm): array
    {
        $packages = $this->packageRepository
            ->createQueryBuilder('package')
            ->where('package.name LIKE :term')
            ->setParameter('term', sprintf('%%%s%%', $searchTerm))
            ->setMaxResults(10)
            ->getQuery()
            ->execute();

        return array_map(
            function (Package $package) {
                return new PackageRecord($package);
            },
            $packages
        );
    }

    public function generateRecordLink(
        Record $record,
        ControllerContext $controllerContext,
        NodeInterface $contextNode,
        bool $absolute
    ): ?string {
        if (!($record instanceof PackageRecord)) {
            $record = $this->findByIdentifier($record->getId());
            if (!$record) {
                return null;
            }
        }

        return $this->packageUrlService->generateUrl($record->getPackage());
    }
}
