<?php declare(strict_types=1);

namespace Newland\Toubiz\Packages\Tests\Unit\LinkHandler;

use Neos\Flow\Mvc\Controller\ControllerContext;
use Neos\Flow\Tests\FunctionalTestCase;
use Newland\NeosCommon\Service\ControllerContextFactory;
use Newland\NeosTestingHelpers\InteractsWithNodes;
use Newland\Toubiz\Packages\Neos\LinkHandler\PackageLinkHandler;
use Newland\Toubiz\Packages\Neos\LinkHandler\PackageRecord;
use Newland\Toubiz\Sync\Neos\Tests\Factory\PackageFactory;

class PackageLinkHandlerTest extends FunctionalTestCase
{
    use InteractsWithNodes;
    protected static $testablePersistenceEnabled = true;

    /** @var PackageLinkHandler */
    protected $subject;

    /** @var PackageFactory */
    protected $packageFactory;

    public function setUp(): void
    {
        parent::setUp();
        $this->subject = $this->objectManager->get(PackageLinkHandler::class);
        $this->packageFactory = new PackageFactory($this->objectManager);
    }

    public function testFindsPackagesBasedOnName(): void
    {
        // Not that second record has lower case 'F'
        $shouldMatch = $this->packageFactory->create([ 'name' => 'Foo Bar' ]);
        $shouldAlsoMatch = $this->packageFactory->create([ 'name' => 'fooooooooBaz' ]);
        $shouldNotMatch = $this->packageFactory->create([ 'name' => 'Baba' ]);

        $results = $this->subject->findBySearchTerm('Foo');
        $ids = array_map(function(PackageRecord $record) {
            return $record->getId();
        }, $results);

        $this->assertContains($shouldMatch->getPersistenceObjectIdentifier(), $ids);
        $this->assertContains($shouldAlsoMatch->getPersistenceObjectIdentifier(), $ids);
        $this->assertNotContains($shouldNotMatch->getPersistenceObjectIdentifier(), $ids);
    }

    public function testFindsPackagesInSpecifiedLanguageOrWithoutLanguage(): void
    {
        $noLang = $this->packageFactory->create([ 'language' => null, 'name' => 'Foo Bar nolang' ]);
        $german = $this->packageFactory->create([ 'language' => 'de', 'name' => 'Foo Bar german' ]);
        $french = $this->packageFactory->create([ 'language' => 'fr', 'name' => 'Foo Bar french' ]);

        $this->subject->setLanguage('fr');
        $results = $this->subject->findBySearchTerm('Foo');
        $ids = array_map(function(PackageRecord $record) {
            return $record->getId();
        }, $results);

        $this->assertContains($noLang->getPersistenceObjectIdentifier(), $ids);
        $this->assertNotContains($german->getPersistenceObjectIdentifier(), $ids);
        $this->assertContains($french->getPersistenceObjectIdentifier(), $ids);
    }

    public function testFindsPackagesBasedOnId(): void
    {
        $package = $this->packageFactory->create();
        $record = $this->subject->findByIdentifier($package->getPersistenceObjectIdentifier());
        $this->assertEquals($package, $record->getPackage());
    }

    public function testGeneratesUrlForPackage(): void
    {
        [ $node, $context ] = $this->nodeAndContext();
        $package = $this->packageFactory->create();
        $record = new PackageRecord($package);
        $url = $this->subject->generateRecordLink($record, $context, $node,true);
        $this->assertNotEmpty($url);
    }

    public function testGeneratesUrlFromRecordFoundByService(): void
    {
        [ $node, $context ] = $this->nodeAndContext();

        // In backend: Editor searches for and saves id of record.
        $this->packageFactory->create([ 'name' => 'foo bar' ]);
        $records = $this->subject->findBySearchTerm('foo');
        $this->assertNotEmpty($records);
        $id = $records[0]->getId();

        // In frontend: id is found in database and then used to generate url.
        $record = $this->subject->findByIdentifier($id);
        $url = $this->subject->generateRecordLink($record, $context, $node,true);

        $this->assertNotEmpty($url);
    }

    private function nodeAndContext(): array
    {
        $node = $this->initializeNode('/sites/foo/bar');
        $context = $this->objectManager->get(ControllerContextFactory::class)->initializeFakeControllerContext($node);
        return [ $node, $context ];
    }
}
