<?php declare(strict_types=1);

namespace Newland\Toubiz\Packages\Neos\LinkHandler;

use Neos\ContentRepository\Domain\Model\NodeInterface;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\I18n\Locale;
use Neos\Flow\I18n\Translator;
use Neos\Flow\Mvc\Controller\ControllerContext;
use Newland\NeosCommon\LinkHandler\Domain\Model\Record;
use Newland\NeosCommon\LinkHandler\Handler\AbstractLinkHandler;
use Newland\Toubiz\Packages\Neos\Service\PackageUrlService;
use Newland\Toubiz\Sync\Neos\Domain\Model\Package;
use Newland\Toubiz\Sync\Neos\Domain\Repository\PackageRepository;

/**
 * @Flow\Scope("singleton")
 */
class PackageLinkHandler extends AbstractLinkHandler
{

    /**
     * @var PackageRepository
     * @Flow\Inject()
     */
    protected $packageRepository;

    /**
     * @var PackageUrlService
     * @Flow\Inject()
     */
    protected $packageUrlService;

    /**
     * @var Translator
     * @Flow\Inject()
     */
    protected $translator;

    public function findByIdentifier(string $recordIdentifier): ?Record
    {
        return $this->identifierToRecord($recordIdentifier);
    }

    public function findBySearchTerm(string $searchTerm): array
    {
        $packages = $this->packageRepository->withLanguage($this->language, function () use ($searchTerm) {
            return $this->packageRepository
                ->createQueryBuilder('package')
                ->where('package.name LIKE :term')
                ->setParameter('term', sprintf('%%%s%%', $searchTerm))
                ->setMaxResults(10)
                ->getQuery()
                ->execute();
        });

        return array_map([ $this, 'packageToRecord' ], $packages);
    }

    public function generateRecordLink(
        Record $record,
        ControllerContext $controllerContext,
        NodeInterface $contextNode,
        bool $absolute
    ): ?string {
        if (!($record instanceof PackageRecord)) {
            $record = $this->identifierToRecord($record->getId());
            if (!$record) {
                return null;
            }
        }

        return $this->packageUrlService->generateUrl($record->getPackage());
    }

    protected function packageToRecord(Package $package): PackageRecord
    {
        $record = new PackageRecord($package);
        $type = $this->translator->translateById(
            'singular',
            [ ],
            null,
            $this->language ? new Locale($this->language) : null,
            'Models/Package',
            'Newland.Toubiz.Packages.Neos'
        );
        if ($type) {
            $record->setTitle(sprintf('%s [%s]', $package->getName(), $type));
        }
        return $record;
    }

    protected function identifierToRecord(string $recordIdentifier): ?PackageRecord
    {
        $package = $this->packageRepository->findByIdentifier($recordIdentifier);
        if (!$package) {
            return null;
        }

        return $this->packageToRecord($package);
    }
}
