<?php
namespace Newland\Toubiz\Packages\Neos\Search\Indexer;

/*
 * This file is part of the "toubiz-packages-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Newland\Contracts\Neos\Search\IndexerInterface;
use Newland\Contracts\Neos\Search\SearchBackend;
use Newland\Contracts\Neos\Search\IndexRecordModification;
use Newland\Contracts\Neos\Search\ProgressHandler;
use Newland\Toubiz\Sync\Neos\Domain\Model\Package;
use Newland\Toubiz\Sync\Neos\Domain\Repository\PackageRepository;

/**
 * Package indexer.
 *
 * Indexes packages for the search.
 *
 * @Flow\Scope("singleton")
 */
class PackageIndexer implements IndexerInterface
{
    public const SCOPE = 'package';

    /**
     * @var PackageRepository
     * @Flow\Inject()
     */
    protected $packageRepository;

    /** @var int|null */
    private $total;

    /** @var string[] */
    private $indexed = [];

    /** @var array */
    protected $configuration;

    public function index(SearchBackend $backend, ProgressHandler $progressHandler): void
    {
        $this->packageRepository->withoutLanguageHandling(function () use ($backend, $progressHandler) {
            $this->total = $this->packageRepository->countAll();

            foreach ($this->packageRepository->findAllIterator() as $packages) {
                foreach ($packages as $package) {
                    $modification = $this->packageToModification($package);
                    $backend->createOrUpdateIndexEntry($modification);
                    $this->indexed[] = $modification->getIdentifier();
                    $progressHandler->informProgress('Packages', \count($this->indexed), $this->total);
                }
            }
        });
    }

    public function postIndex(SearchBackend $backend): void
    {
        $backend->deleteObsoleteIndexEntries($this->indexed);
    }

    public function setConfiguration(array $configuration = []): void
    {
        $this->configuration = $configuration;
    }

    private function packageToModification(Package $package): IndexRecordModification
    {
        $content = [
            $package->getName(),
        ];

        return new IndexRecordModification(
            (string) $package->getPersistenceObjectIdentifier(),
            $package->getName(),
            '',
            trim(implode(' ', $content)),
            static::SCOPE,
            $package->getLanguage()
        );
    }
}
