<?php
namespace Newland\Toubiz\Packages\Neos\Controller;

/*
 * This file is part of the "toubiz-packages-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\ContentRepository\Domain\Model\Node;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\I18n\Locale;
use Neos\Flow\I18n\Service;
use Neos\FluidAdaptor\View\StandaloneView;
use Neos\Flow\Mvc\Controller\ActionController;
use Neos\Flow\Mvc\View\ViewInterface;
use Neos\Neos\Domain\Model\Site;
use Neos\Neos\Domain\Service\ContentContext;

/**
 * Abstract action controller.
 *
 * @Flow\Scope("singleton")
 */
abstract class AbstractActionController extends ActionController
{
    /**
     * The configuration for this package.
     *
     * @Flow\InjectConfiguration
     * @var array
     */
    protected $configuration;

    /**
     * The configuration for the sync package.
     *
     * @Flow\InjectConfiguration(package="Newland.Toubiz.Sync.Neos")
     * @var array
     */
    protected $syncConfiguration;

    /**
     * @var Node The current node.
     */
    protected $node;

    /**
     * @var Site
     */
    protected $currentSite;

    /**
     * @var array
     */
    protected $properties;

    /**
     * @Flow\Inject
     * @var Service
     */
    protected $i18nService;

    /**
     * Action initializer.
     *
     * Called before the called action. This may be called
     * manually inside an action that is working outside of
     * the default node/context, like an AJAX action.
     *
     * @param Node|null $node
     * @return void
     */
    protected function initializeAction(Node $node = null)
    {
        $node = $node ?? $this->currentNode();
        if ($node === null) {
            return;
        }
        $this->node = $node;

        $this->properties = (array) $this->node->getProperties();

        $context = $this->node->getContext();
        if ($context instanceof ContentContext) {
            $this->currentSite = $context->getCurrentSite();
        }
    }

    /**
     * View initializer.
     *
     * Called before a view is rendered.
     *
     * @param ViewInterface $view
     * @return void
     */
    protected function initializeView(ViewInterface $view)
    {
        $view->assignMultiple(
            [
                'configuration' => $this->configuration,
                'node' => $this->node,
            ]
        );
    }

    /**
     * Creates a fluid standalone view.
     *
     * @param string $template Path to the template to render.
     * @return StandaloneView
     */
    protected function createStandaloneView($template)
    {
        $view = new StandaloneView;
        $this->initializeView($view);
        $view->setTemplateRootPaths([ 'resource://Newland.Toubiz.Packages.Neos/Private/Templates' ]);
        $view->setPartialRootPaths([ 'resource://Newland.Toubiz.Packages.Neos/Private/Partials' ]);
        $view->setTemplate($template);

        $dimensions = $this->node->getContext()->getDimensions();
        if (array_key_exists('language', $dimensions) && $dimensions['language'] !== []) {
            $currentLocale = new Locale($dimensions['language'][0]);
            $this->i18nService->getConfiguration()->setCurrentLocale($currentLocale);
            $this->i18nService->getConfiguration()->setFallbackRule(
                [
                    'strict' => false,
                    'order' => array_reverse($dimensions['language']),
                ]
            );
        }

        return $view;
    }

    private function currentNode(): ?Node
    {
        /** @var Node|null $node */
        $node = $this->request->getInternalArgument('__node');
        return ($node instanceof Node) ? $node : null;
    }
}
