<?php declare(strict_types=1);

namespace Newland\Toubiz\Offer\Neos\LinkHandler;

use Neos\ContentRepository\Domain\Model\NodeInterface;
use Neos\Flow\I18n\Locale;
use Neos\Flow\I18n\Translator;
use Neos\Flow\Mvc\Controller\ControllerContext;
use Newland\NeosCommon\LinkHandler\Domain\Model\Record;
use Newland\NeosCommon\LinkHandler\Handler\AbstractLinkHandler;
use Newland\Toubiz\Offer\Neos\Service\OfferUrlService;
use Newland\Toubiz\Sync\Neos\Domain\Model\Offer;
use Newland\Toubiz\Sync\Neos\Domain\Repository\OfferRepository;
use Neos\Flow\Annotations as Flow;

/**
 * @Flow\Scope("singleton")
 */
class OfferLinkHandler extends AbstractLinkHandler
{

    /**
     * @var OfferRepository
     * @Flow\Inject()
     */
    protected $offerRepository;

    /**
     * @var OfferUrlService
     * @Flow\Inject()
     */
    protected $offerUrlService;

    /**
     * @var Translator
     * @Flow\Inject()
     */
    protected $translator;

    public function findByIdentifier(string $recordIdentifier): ?Record
    {
        return $this->identifierToRecord($recordIdentifier);
    }

    public function findBySearchTerm(string $searchTerm): array
    {
        $offers = $this->offerRepository->withLanguage($this->language, function () use ($searchTerm) {
            return $this->offerRepository
                ->createQueryBuilder('offer')
                ->where('offer.name LIKE :term')
                ->setParameter('term', '%' . $searchTerm . '%')
                ->setMaxResults(10)
                ->getQuery()
                ->execute();
        });

        return array_map([ $this, 'offerToRecord' ], $offers);
    }

    public function generateRecordLink(
        Record $record,
        ControllerContext $controllerContext,
        NodeInterface $contextNode,
        bool $absolute
    ): ?string {
        if (!($record instanceof OfferRecord)) {
            $record = $this->identifierToRecord($record->getId());
            if (!$record) {
                return null;
            }
        }

        return $this->offerUrlService->generateUrl($record->getOffer());
    }

    protected function offerToRecord(Offer $offer): OfferRecord
    {
        $record = new OfferRecord($offer);
        $type = $this->translator->translateById(
            'singular',
            [ ],
            null,
            $this->language ? new Locale($this->language) : null,
            'Models/Offer',
            'Newland.Toubiz.Offer.Neos'
        );
        if ($type) {
            $record->setTitle(sprintf('%s [%s]', $offer->getName(), $type));
        }
        return $record;
    }

    protected function identifierToRecord(string $identifier): ?OfferRecord
    {
        $offer = $this->offerRepository->findByIdentifier($identifier);
        if (!$offer) {
            return null;
        }

        return $this->offerToRecord($offer);
    }
}
