<?php declare(strict_types=1);

namespace Newland\Toubiz\Offer\Tests\Unit\LinkHandler;

use Neos\Flow\Tests\FunctionalTestCase;
use Newland\NeosCommon\Service\ControllerContextFactory;
use Newland\NeosTestingHelpers\InteractsWithNodes;
use Newland\Toubiz\Offer\Neos\LinkHandler\OfferLinkHandler;
use Newland\Toubiz\Offer\Neos\LinkHandler\OfferRecord;
use Newland\Toubiz\Sync\Neos\Tests\Factory\OfferFactory;

class OfferLinkHandlerTest extends FunctionalTestCase
{
    use InteractsWithNodes;
    protected static $testablePersistenceEnabled = true;

    /** @var OfferLinkHandler */
    protected $subject;

    /** @var OfferFactory */
    protected $offerFactory;

    public function setUp(): void
    {
        parent::setUp();
        $this->subject = $this->objectManager->get(OfferLinkHandler::class);
        $this->offerFactory = new OfferFactory($this->objectManager);
    }

    public function testFindsOffersBasedOnName(): void
    {
        // Not that second record has lower case 'F'
        $shouldMatch = $this->offerFactory->create([ 'name' => 'Foo Bar' ]);
        $shouldAlsoMatch = $this->offerFactory->create([ 'name' => 'fooooooooBaz' ]);
        $shouldNotMatch = $this->offerFactory->create([ 'name' => 'Baba' ]);

        $results = $this->subject->findBySearchTerm('Foo');
        $ids = array_map(function(OfferRecord $record) {
            return $record->getId();
        }, $results);

        $this->assertContains($shouldMatch->getPersistenceObjectIdentifier(), $ids);
        $this->assertContains($shouldAlsoMatch->getPersistenceObjectIdentifier(), $ids);
        $this->assertNotContains($shouldNotMatch->getPersistenceObjectIdentifier(), $ids);
    }

    public function testFindsOffersBasedOnId(): void
    {
        $offer = $this->offerFactory->create();
        $record = $this->subject->findByIdentifier($offer->getPersistenceObjectIdentifier());
        $this->assertEquals($offer, $record->getOffer());
    }

    public function testGeneratesUrlForOffer(): void
    {
        [ $node, $context ] = $this->nodeAndContext();
        $offer = $this->offerFactory->create();
        $record = new OfferRecord($offer);
        $url = $this->subject->generateRecordLink($record, $context, $node,true);
        $this->assertNotEmpty($url);
    }

    public function testGeneratesUrlFromRecordFoundByService(): void
    {
        [ $node, $context ] = $this->nodeAndContext();

        // In backend: Editor searches for and saves id of record.
        $this->offerFactory->create([ 'name' => 'foo bar' ]);
        $records = $this->subject->findBySearchTerm('foo');
        $this->assertNotEmpty($records);
        $id = $records[0]->getId();

        // In frontend: id is found in database and then used to generate url.
        $record = $this->subject->findByIdentifier($id);
        $url = $this->subject->generateRecordLink($record, $context, $node,true);

        $this->assertNotEmpty($url);
    }

    private function nodeAndContext(): array
    {
        $node = $this->initializeNode('/sites/foo/bar');
        $context = $this->objectManager->get(ControllerContextFactory::class)->initializeFakeControllerContext($node);
        return [ $node, $context ];
    }
}
