<?php
namespace Newland\Toubiz\Offer\Neos\Search\Indexer;

/*
 * This file is part of the "toubiz-offer-neos" offer.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Newland\Contracts\Neos\Search\IndexerInterface;
use Newland\Contracts\Neos\Search\SearchBackend;
use Newland\Contracts\Neos\Search\IndexRecordModification;
use Newland\Contracts\Neos\Search\ProgressHandler;
use Newland\Toubiz\Sync\Neos\Domain\Model\Offer;
use Newland\Toubiz\Sync\Neos\Domain\Repository\OfferRepository;

/**
 * Offer indexer.
 *
 * Indexes offers for the search.
 *
 * @Flow\Scope("singleton")
 */
class OfferIndexer implements IndexerInterface
{
    public const SCOPE = 'offer';

    /**
     * @var OfferRepository
     * @Flow\Inject()
     */
    protected $offerRepository;

    /** @var int|null */
    private $total;

    /** @var string[] */
    private $indexed = [];

    /** @var array */
    protected $configuration;

    public function index(SearchBackend $backend, ProgressHandler $progressHandler): void
    {
        $this->offerRepository->withoutLanguageHandling(function () use ($backend, $progressHandler) {
            $this->total = $this->offerRepository->countAll();

            foreach ($this->offerRepository->findAllIterator() as $offers) {
                foreach ($offers as $offer) {
                    $modification = $this->offerToModification($offer);
                    $backend->createOrUpdateIndexEntry($modification);
                    $this->indexed[] = $modification->getIdentifier();
                    $progressHandler->informProgress('Offers', \count($this->indexed), $this->total);
                }
            }
        });
    }

    public function postIndex(SearchBackend $backend): void
    {
        $backend->deleteObsoleteIndexEntries($this->indexed);
    }

    public function setConfiguration(array $configuration = []): void
    {
        $this->configuration = $configuration;
    }

    private function offerToModification(Offer $offer): IndexRecordModification
    {
        $content = [
            $offer->getName(),
        ];

        return new IndexRecordModification(
            (string) $offer->getPersistenceObjectIdentifier(),
            $offer->getName(),
            '',
            trim(implode(' ', $content)),
            static::SCOPE,
            $offer->getLanguage()
        );
    }
}
