<?php declare(strict_types=1);

namespace Newland\Toubiz\Map\Neos\Serialization\Markers;

use Neos\ContentRepository\Domain\Model\NodeInterface;
use Neos\ContentRepository\Domain\Projection\Content\TraversableNodeInterface;
use Neos\Flow\Configuration\Exception\InvalidConfigurationException;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Poi\Neos\Filter\ArticleFilterFactory;
use Newland\Toubiz\Sync\Neos\Domain\Model\Article;
use Neos\Flow\Annotations as Flow;

/**
 * @Flow\Scope("singleton")
 */
class FilteredArticles implements NodeSerializer
{
    use ArticleFormatting;
    protected const NODE_TYPE = 'Newland.Toubiz.Map.Neos:Map.Markers.FilteredArticles';

    /**
     * @var int
     * @Flow\InjectConfiguration(package="Newland.Toubiz.Map.Neos", path="filteredArticles.limit")
     */
    protected $limit;

    public function canSerialize(TraversableNodeInterface $node): bool
    {
        return $node->getNodeType()->isOfType(static::NODE_TYPE);
    }

    public function serialize(TraversableNodeInterface $node): array
    {
        if (!$node->hasProperty('articleType')) {
            return [];
        }

        $mainType = (int) $node->getProperty('articleType');
        if (!\in_array($mainType, ArticleConstants::ALL_TYPES, true)) {
            throw new InvalidConfigurationException(
                sprintf(
                    '"%s" is not a valid article type. Must be one of [%s]',
                    $node->getProperty('articleType'),
                    implode(', ', ArticleConstants::ALL_TYPES)
                )
            );
        }

        if (!($node instanceof NodeInterface)) {
            return [];
        }

        /** @var Article[] $articles */
        $articles = (new ArticleFilterFactory($node))
            ->createFilterForArticleType($mainType, (array) $node->getProperties())
            ->setPageSize($this->limit)
            ->getArticleQuery()
            ->getQuery()
            ->execute();

        $alwaysDisplayPolygon = (bool) $node->getProperty('alwaysDisplayPolygon');
        $color = $node->getProperty('tourColor');
        $markers = [];
        foreach ($articles as $article) {
            $markers[$article->getPersistenceObjectIdentifier()]  = $this->articleToMarker($article, $node, [
                'tour' => [
                    'alwaysDisplayPolygon' => $alwaysDisplayPolygon,
                    'color' => $color,
                ]
            ]);
        }

        return $markers;
    }
}
