<?php
namespace Newland\Toubiz\Map\Neos\Service;

use Doctrine\ORM\Query;
use Doctrine\ORM\QueryBuilder;
use Neos\ContentRepository\Domain\Model\Node;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Cache\CacheManager;
use Neos\Flow\Mvc\ActionRequest;
use Neos\Flow\Mvc\Controller\ControllerContext;
use Neos\Flow\Persistence\Doctrine\Mapping\ClassMetadata;
use Newland\Toubiz\Poi\Neos\Service\ArticleUrlService;
use Newland\Toubiz\Map\Neos\DependencyInjection\SoftInject;
use Newland\Toubiz\Sync\Neos\Domain\Model\Article;
use Newland\Toubiz\Sync\Neos\Domain\Model\Category;
use function Safe\json_encode;

/**
 * @Flow\Scope("singleton")
 */
class ArticleCacheService
{
    const CACHE_NAME = 'Newland_Toubiz_Map_Neos-Data';
    const MAX_ARTICLES_TO_CACHE = 2000;

    /**
     * @Flow\Inject
     * @var CacheManager
     */
    protected $cacheManager;

    /**
     * @SoftInject(errorMessage="Package Newland.Toubiz.Poi.Neos is required for this action.")
     * @var ArticleUrlService
     */
    protected $articleUrl;

    public function cacheQueryResultsAsJson(QueryBuilder $query, ControllerContext $context): string
    {
        return $this->generateWithoutCachingIfNotPublicContext($query, $context) ?:
            $this->fetchFromCache($query, $context) ?:
            $this->generateAndStoreInCache($query, $context);
    }

    public function generateAndStoreInCache(QueryBuilder $query, ControllerContext $context): string
    {
        $cache = $this->cacheManager->getCache(static::CACHE_NAME);
        $json = $this->generate($query, $context);
        $cache->set($this->cacheKey($query, $context), $json);
        return $json;
    }

    private function generateWithoutCachingIfNotPublicContext(QueryBuilder $query, ControllerContext $context)
    {
        $node = $this->nodeFromContext($context);
        if ($node === null) {
            return null;
        }

        if ($node->getContext()->getWorkspace()->isPublicWorkspace()) {
            return null;
        }

        return $this->generate($query, $context);
    }


    private function fetchFromCache(QueryBuilder $query, ControllerContext $context)
    {
        $cache = $this->cacheManager->getCache(static::CACHE_NAME);
        $cacheKey = $this->cacheKey($query, $context);

        if ($cache->has($cacheKey)) {
            return $cache->get($cacheKey);
        }
        return null;
    }


    private function cacheKey(QueryBuilder $query, ControllerContext $context): string
    {
        $node = $this->nodeFromContext($context);
        if (!$node) {
            return '';
        }

        return md5($node->getPath() . json_encode($node->getDimensions()) . $query->getDQL());
    }

    private function generate(QueryBuilder $query, ControllerContext $context): string
    {
        $cached = [];

        $query = $query->getQuery()
            ->setFirstResult(0)
            ->setFetchMode(Article::class, 'categories', ClassMetadata::FETCH_EAGER)
            ->setFetchMode(Article::class, 'media', ClassMetadata::FETCH_EAGER)
            ->setFetchMode(Category::class, 'articles', ClassMetadata::FETCH_EXTRA_LAZY);

        $this->paginate(
            $query,
            500,
            static::MAX_ARTICLES_TO_CACHE,
            function (array &$articles) use (&$cached, &$context, &$query) {
                foreach ($articles as $article) {
                    /** @var Article $article */
                    $serialized = $article->jsonSerialize();
                    $serialized['url'] = $this->articleUrl->generateUrl($article, $context);
                    $cached[] = $serialized;
                }

                $query->getEntityManager()->clear();
                \gc_collect_cycles();
            }
        );

        return json_encode($cached);
    }

    private function paginate(Query $query, int $pageSize, int $maxItems, callable $callback)
    {
        $page = 0;
        do {
            $itemsOnThisPage = min($pageSize, $maxItems - ($page * $pageSize));
            $itemsOnThisPage = max(0, $itemsOnThisPage);

            $results = $query
                ->setFirstResult($page * $pageSize)
                ->setMaxResults($itemsOnThisPage)
                ->execute();

            $callback($results);
            $page++;
        } while ($results && count($results) > 0);
    }


    /**
     * @param ControllerContext $context
     * @return Node|null
     */
    private function nodeFromContext(ControllerContext $context)
    {
        $request = $context->getRequest();
        if (!($request instanceof ActionRequest)) {
            return null;
        }

        /** @var Node|null $node */
        $node = $request->getInternalArgument('__node');
        if (!($node instanceof Node)) {
            return null;
        }

        return $node;
    }
}
