<?php declare(strict_types=1);

namespace Newland\Toubiz\Map\Neos\Serialization;

use Neos\ContentRepository\Domain\Factory\NodeFactory;
use Neos\ContentRepository\Domain\Model\NodeData;
use Neos\ContentRepository\Domain\Model\Workspace;
use Neos\ContentRepository\Domain\Projection\Content\NodeInterface;
use Neos\ContentRepository\Domain\Projection\Content\TraversableNodeInterface;
use Neos\ContentRepository\Domain\Repository\NodeDataRepository;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Persistence\PersistenceManagerInterface;
use Neos\Neos\Domain\Service\ContentContextFactory;

/**
 * Slot that forces regeneration of the map serialization cache every time the nodes are changed.
 *
 * @Flow\Scope("singleton")
 */
class NodeSlot
{

    /**
     * @var SerializationService
     * @Flow\Inject()
     */
    protected $serializationService;

    /**
     * @var NodeDataRepository
     * @Flow\Inject()
     */
    protected $nodeDataRepository;

    /**
     * @var NodeFactory
     * @Flow\Inject()
     */
    protected $nodeFactory;

    /**
     * @var ContentContextFactory
     * @Flow\Inject()
     */
    protected $contentContextFactory;

    /**
     * @var PersistenceManagerInterface
     * @Flow\Inject()
     */
    protected $persistenceManager;

    private function initialize(): void
    {
        // New nodes may not be in db yet.
        $this->persistenceManager->persistAll();
    }

    public function onNodePropertyChanged(TraversableNodeInterface $node): void
    {
        $this->initialize();
        if (!$node->getNodeType()->isOfType(MapSerializer::TYPE_MARKERS_WILDCARD)) {
            return;
        }

        $this->regenerate($node);
    }

    public function onNodePathChanged(TraversableNodeInterface $node): void
    {
        $this->initialize();
        if (!$node->getNodeType()->isOfType(MapSerializer::TYPE_FILTER_ITEM)) {
            return;
        }

        $this->regenerate($node);
    }

    public function onNodePublished(NodeInterface $node, Workspace $targetWorkspace): void
    {
        $this->initialize();
        $mapNode = $this->getMapNode($node);
        $nodeInTargetWorkspace = $this->fetchNodeInGivenWorkspace($mapNode, $targetWorkspace);
        if ($nodeInTargetWorkspace === null) {
            return;
        }
        $this->regenerate($nodeInTargetWorkspace);
    }

    private function regenerate(TraversableNodeInterface $node): void
    {
        $this->serializationService->forceRegenerateAllCacheEntries($node);

        /*
         * Currently only the cache on the map node directly is used in the frontend.
         * Therefor the cache on the filter item & marker nodes are not warmed.
         * If that changes the `forceRegenerateAllCacheEntriesOfSubNodes` should be called here.
         */
    }

    private function getParentNodeOfType(NodeInterface $node, string $type): ?NodeInterface
    {
        if (!($node instanceof TraversableNodeInterface)) {
            return null;
        }

        do {
            if ($node->getNodeType()->isOfType($type)) {
                return $node;
            }
            // phpcs:ignore
        } while ($node = $node->findParentNode());

        return null;
    }

    private function getMapNode(NodeInterface $node): ?TraversableNodeInterface
    {
        if (!$this->isOfOneOfGivenTypes($node, MapSerializer::ALL_TYPES)) {
            return null;
        }

        $mapNode = $this->getParentNodeOfType($node, MapSerializer::TYPE_MAP);
        if ($mapNode === null || !($mapNode instanceof TraversableNodeInterface)) {
            return null;
        }

        return $mapNode;
    }

    /** @param string[] $types */
    protected function isOfOneOfGivenTypes(NodeInterface $node, array $types): bool
    {
        foreach ($types as $type) {
            if ($node->getNodeType()->isOfType($type)) {
                return true;
            }
        }
        return false;
    }

    private function fetchNodeInGivenWorkspace(
        ?TraversableNodeInterface $node,
        Workspace $workspace
    ): ?TraversableNodeInterface {

        if ($node === null) {
            return null;
        }

        /** @var NodeData|null $data */
        $data = $this->nodeDataRepository->findOneByPath((string) $node->findNodePath(), $workspace);
        if ($data === null) {
            return null;
        }

        $nodeInTargetWorkspace = $this->nodeFactory->createFromNodeData(
            $data,
            $this->contentContextFactory->create(
                [
                    'workspaceName' => $data->getWorkspace()->getName(),
                    'invisibleContentShown' => false,
                    'inaccessibleContentShown' => false,
                    'removedContentShown' => false,
                    'dimensions' => $data->getDimensionValues(),
                ]
            )
        );

        if (!($nodeInTargetWorkspace instanceof TraversableNodeInterface)) {
            return null;
        }

        return $nodeInTargetWorkspace;
    }
}
