<?php

namespace Newland\Toubiz\Map\Neos\Tests\Unit\Serialization;

use Neos\ContentRepository\Domain\Model\Node;
use Neos\Flow\Tests\FunctionalTestCase;
use Neos\Neos\Domain\Model\Site;
use Newland\NeosTestingHelpers\InteractsWithNodes;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Map\Neos\Serialization\MapSerializer;
use Newland\Toubiz\Sync\Neos\Tests\Factory\ArticleFactory;
use Newland\Toubiz\Sync\Neos\Tests\Factory\AttributeFactory;
use Newland\Toubiz\Sync\Neos\Tests\Factory\CategoryFactory;

class MapSerializerTest extends FunctionalTestCase
{
    protected static $testablePersistenceEnabled = true;
    use InteractsWithNodes;

    /** @var MapSerializer */
    protected $subject;

    /** @var ArticleFactory */
    protected $articleFactory;

    /** @var AttributeFactory */
    protected $attributeFactory;

    /** @var CategoryFactory */
    protected $categoryFactory;

    /** @var Node[] */
    protected $nodes;

    /** @var Site */
    protected $site;

    public function setUp(): void
    {
        parent::setUp();
        $this->subject = $this->objectManager->get(MapSerializer::class);
        $this->articleFactory = new ArticleFactory($this->objectManager);
        $this->attributeFactory = new AttributeFactory($this->objectManager);
        $this->categoryFactory = new CategoryFactory($this->objectManager);

        $this->site = $this->initializeSite('foo');
        $this->nodes = $this->nodes(
            [
                '/sites/foo' => [ 'nodeType' => 'Neos.NodeTypes:Page' ],
                '/sites/foo/main' => [ 'nodeType' => 'Neos.Neos:ContentCollection' ],
                '/sites/foo/main/map' => [ 'nodeType' => 'Newland.Toubiz.Map.Neos:Map' ],
                '/sites/foo/main/map/children' => [ 'nodeType' => 'Neos.Neos:ContentCollection' ],
                '/sites/foo/main/map/children/first-filter-item' => [
                    'nodeType' => 'Newland.Toubiz.Map.Neos:Map.FilterItem',
                    'properties' => [ 'title' => 'First Title' ],
                ],
                '/sites/foo/main/map/children/first-filter-item/children' => [ 'nodeType' => 'Neos.Neos:ContentCollection' ],
                '/sites/foo/main/map/children/second-filter-item' => [
                    'nodeType' => 'Newland.Toubiz.Map.Neos:Map.FilterItem',
                    'properties' => [ 'title' => 'Second Title' ],
                ],
                '/sites/foo/main/map/children/second-filter-item/children' => [ 'nodeType' => 'Neos.Neos:ContentCollection' ],
                '/sites/foo/main/map/children/first-filter-item/children/deep-filter-item' => [
                    'nodeType' => 'Newland.Toubiz.Map.Neos:Map.FilterItem',
                    'properties' => [ 'title' => 'Deep Title' ],
                ],
                '/sites/foo/main/map/children/first-filter-item/children/deep-filter-item/children' => [ 'nodeType' => 'Neos.Neos:ContentCollection' ],
            ],
            [
                'site' => $this->site,
                'workspaceName' => 'live',
            ]
        );
    }

    public function testAddsChildrenContainersAsFilterItems(): void
    {
        $firstId = $this->nodes['/sites/foo/main/map/children/first-filter-item']->getIdentifier();
        $secondId = $this->nodes['/sites/foo/main/map/children/second-filter-item']->getIdentifier();
        $deepId = $this->nodes['/sites/foo/main/map/children/first-filter-item/children/deep-filter-item']->getIdentifier();

        $result = $this->subject->serializeFilterItems($this->nodes['/sites/foo/main/map']);

        $this->assertArrayHasKey($firstId, $result, 'Should have array item for first container');
        $this->assertEquals(
            '/sites/foo/main/map/children/first-filter-item',
            $result[$firstId]['path'],
            'Should have path of first container'
        );
        $this->assertEquals('First Title', $result[$firstId]['title'], 'Should have title of first container');

        $this->assertArrayHasKey($secondId, $result, 'Should have array item for second container');
        $this->assertEquals(
            '/sites/foo/main/map/children/second-filter-item',
            $result[$secondId]['path'],
            'Should have path of second container'
        );
        $this->assertEquals('Second Title', $result[$secondId]['title'], 'Should have title of second container');

        $this->assertArrayHasKey($deepId, $result[$firstId]['children'], 'Should have array item for deep container');
        $this->assertEquals(
            '/sites/foo/main/map/children/first-filter-item/children/deep-filter-item',
            $result[$firstId]['children'][$deepId]['path'],
            'Should have path of deep container'
        );
        $this->assertEquals(
            'Deep Title',
            $result[$firstId]['children'][$deepId]['title'],
            'Should have title of deep container'
        );
    }

    public function testIgnoresHiddenFilterItems(): void
    {
        $this->nodes['/sites/foo/main/map/children/first-filter-item']->setHidden(true);
        $this->persistNode($this->nodes['/sites/foo/main/map/children/first-filter-item']);
        $firstId = $this->nodes['/sites/foo/main/map/children/first-filter-item']->getIdentifier();
        $childId = $this->nodes['/sites/foo/main/map/children/first-filter-item/children/deep-filter-item']->getIdentifier();

        $result = $this->subject->serializeFilterItems($this->nodes['/sites/foo/main/map']);
        $this->assertArrayNotHasKey($firstId, $result);
        $this->assertArrayNotHasKey($childId, $result);
    }

    public function testSerializesArticles(): void
    {
        $this->articleFactory->createMultiple(7, [ 'mainType' => ArticleConstants::TYPE_TOUR ]);
        $this->articleFactory->createMultiple(9, [ 'mainType' => ArticleConstants::TYPE_ATTRACTION ]);
        $this->initializeNode(
            '/sites/foo/main/map/children/first-filter-item/children/filtered-articles',
            'live',
            $this->site,
            [ 'articleType' => ArticleConstants::TYPE_TOUR ],
            'de',
            'Newland.Toubiz.Map.Neos:Map.Markers.FilteredArticles'
        );

        $result = $this->subject->serializeMarkers($this->nodes['/sites/foo/main/map']);
        $this->assertCount(7, $result);
    }

    public function testSerializesSelectedArticles(): void
    {
        $articles = $this->articleFactory->createMultiple(5);
        $this->initializeNode(
            '/sites/foo/main/map/children/first-filter-item/children/selected-articles',
            'live',
            $this->site,
            [
                'articles' => [
                    $articles[0]->getPersistenceObjectIdentifier(),
                    $articles[1]->getPersistenceObjectIdentifier(),
                    $articles[2]->getPersistenceObjectIdentifier(),
                ]
            ],
            'de',
            'Newland.Toubiz.Map.Neos:Map.Markers.SelectedArticles'
        );

        $result = $this->subject->serializeMarkers($this->nodes['/sites/foo/main/map']);
        $this->assertCount(3, $result);
    }

    public function testSerializesLink(): void
    {
        $this->initializeNode(
            '/sites/foo/main/map/children/first-filter-item/children/link',
            'live',
            $this->site,
            [
                'uri' => 'https://land-in-sicht.de',
                'title' => 'Land in Sicht',
                'categoryTitle' => 'Piraten',
                'description' => 'An agency',
                'latitude' => 47.980869,
                'longitude' => 7.820810,
            ],
            'de',
            'Newland.Toubiz.Map.Neos:Map.Markers.Link'
        );

        $result = $this->subject->serializeMarkers($this->nodes['/sites/foo/main/map']);
        $this->assertCount(1, $result);
    }

    public function testIgnoresMarkersOfHiddenFilterItems(): void
    {
        $articles = $this->articleFactory->createMultiple(5);
        $this->initializeNode(
            '/sites/foo/main/map/children/first-filter-item/children/selected-articles',
            'live',
            $this->site,
            [
                'articles' => [
                    $articles[0]->getPersistenceObjectIdentifier(),
                    $articles[1]->getPersistenceObjectIdentifier(),
                    $articles[2]->getPersistenceObjectIdentifier(),
                ]
            ],
            'de',
            'Newland.Toubiz.Map.Neos:Map.Markers.SelectedArticles'
        );

        $this->nodes['/sites/foo/main/map/children/first-filter-item']->setHidden(true);
        $this->persistNode($this->nodes['/sites/foo/main/map/children/first-filter-item']);

        $result = $this->subject->serializeMarkers($this->nodes['/sites/foo/main/map']);
        $this->assertCount(0, $result);
    }

    public function testIgnoresMarkersOfChildrenOfHiddenFilterItems(): void
    {
        $articles = $this->articleFactory->createMultiple(5);
        $this->initializeNode(
            '/sites/foo/main/map/children/first-filter-item/children/deep-filter-item/children/selected-articles',
            'live',
            $this->site,
            [
                'articles' => [
                    $articles[0]->getPersistenceObjectIdentifier(),
                    $articles[1]->getPersistenceObjectIdentifier(),
                    $articles[2]->getPersistenceObjectIdentifier(),
                ]
            ],
            'de',
            'Newland.Toubiz.Map.Neos:Map.Markers.SelectedArticles'
        );

        $this->nodes['/sites/foo/main/map/children/first-filter-item']->setHidden(true);
        $this->persistNode($this->nodes['/sites/foo/main/map/children/first-filter-item']);

        $result = $this->subject->serializeMarkers($this->nodes['/sites/foo/main/map']);
        $this->assertCount(0, $result);
    }

    public function testOnlyASingleFilterItemCanBeOpenedByDefault(): void
    {
        $first = $this->nodes['/sites/foo/main/map/children/first-filter-item'];
        $second = $this->nodes['/sites/foo/main/map/children/second-filter-item'];
        $first->setProperty('openByDefault', true);
        $this->persistNode($first);
        $second->setProperty('openByDefault', true);
        $this->persistNode($second);

        $result = $this->subject->serializeFilterItems($this->nodes['/sites/foo/main/map']);
        $this->assertTrue($result[$first->getIdentifier()]['openByDefault']);
        $this->assertFalse($result[$second->getIdentifier()]['openByDefault']);
    }

}
