<?php declare(strict_types=1);
namespace Newland\Toubiz\Map\Neos\ViewHelpers;

use Neos\ContentRepository\Domain\Model\Node;
use Neos\ContentRepository\Domain\Model\NodeInterface;
use Neos\Flow\Configuration\ConfigurationManager;
use Neos\Flow\Mvc\ActionRequest;
use Neos\FluidAdaptor\Core\ViewHelper\AbstractTagBasedViewHelper;
use Neos\Flow\Annotations as Flow;
use Newland\NeosIcons\IconUrlService;
use Newland\Toubiz\Map\Neos\Service\ApiKeyService;
use function Safe\json_encode;

class StaticMapViewHelper extends AbstractTagBasedViewHelper
{
    /**
     * @var ConfigurationManager
     * @Flow\Inject()
     */
    protected $configurationManager;

    /**
     * @var array
     * @Flow\InjectConfiguration(package="Newland.Toubiz.Map.Neos", path="geolocation")
     */
    protected $geolocationSettings;

    /**
     * @var ApiKeyService
     * @Flow\Inject()
     */
    protected $apiKeyService;

    public function render(
        float $latitude,
        float $longitude,
        string $configurationPath,
        NodeInterface $node,
        int $zoom = null,
        string $mapType = null,
        ?array $geometry = null
    ): string {
        $staticMapConfiguration = (array) $this->configurationManager->getConfiguration(
            ConfigurationManager::CONFIGURATION_TYPE_SETTINGS,
            $configurationPath
        );

        $mapType = $mapType ?? $staticMapConfiguration['mapType'] ?? null;
        $staticMapType = $staticMapConfiguration['staticMapType'] ?? null;
        $dynamicMapType = $staticMapConfiguration['dynamicMapType'] ?? null;

        $geolocationEnabled = $this->geolocationSettings['enabled'] ?? false;

        $zoom = $zoom ?? $staticMapConfiguration['zoom'] ?? 75;
        if ($mapType === null) {
            return '<!-- Static map not rendered: No map type defined. -->';
        }
        $this->addApiKeysToTags($node);
        $this->addSourceSetAttributes($staticMapConfiguration);
        $this->tag->addAttributes(
            [
                'class' => 'cursor-pointer',
                'data-toubiz-standalone-map' => '',
                'data-toubiz-standalone-map.map-type' => $mapType,
                'data-toubiz-standalone-map.map.static-map-type' => $staticMapType,
                'data-toubiz-standalone-map.map.dynamic-map-type' => $dynamicMapType,
                'data-toubiz-standalone-map.map.zoom' => $zoom,
                'data-toubiz-standalone-map.map.geolocation.enabled' => '(bool) ' . $geolocationEnabled,
                'data-toubiz-standalone-map.map.geolocation.wait-for-user-intent' => '(bool) true',
                'data-toubiz-standalone-map.markers.0.latitude' => $latitude,
                'data-toubiz-standalone-map.markers.0.longitude' => $longitude,
            ]
        );

        if ($geometry) {
            $this->tag->addAttribute('data-toubiz-standalone-map.markers.0.tour', json_encode($geometry));
        }

        $iconUrl = $staticMapConfiguration['markerIcon']['url'] ?? null;
        if ($iconUrl) {
            $iconWidth = $staticMapConfiguration['markerIcon']['width'] ?? 64;
            $iconHeight = $staticMapConfiguration['markerIcon']['height'] ?? 64;
            $this->tag->addAttributes([
                'data-toubiz-standalone-map.map.static-marker-icon-url' => $iconUrl,
                'data-toubiz-standalone-map.markers.0.icon-url' => $iconUrl,
                'data-toubiz-standalone-map.markers.0.icon-width' => '(int)' . $iconWidth,
                'data-toubiz-standalone-map.markers.0.icon-height' => '(int)' . $iconHeight,
            ]);
        }

        $this->tag->forceClosingTag(true);
        return $this->tag->render();
    }
    private function addApiKeysToTags(NodeInterface $node): void
    {
        $apiKeys = $this->apiKeyService->getApiKeys($node);
        if (array_key_exists('googleMaps', $apiKeys)) {
            $this->tag->addAttribute(
                'data-toubiz-standalone-map.map.google-maps-api-key',
                $apiKeys['googleMaps']
            );
        }
        if (array_key_exists('toursprung', $apiKeys)) {
            $this->tag->addAttribute(
                'data-toubiz-standalone-map.map.toursprung-api-key',
                $apiKeys['toursprung']
            );
        }
    }

    private function addSourceSetAttributes(array $staticMapConfiguration): void
    {
        foreach ($staticMapConfiguration['srcSet'] ?? [] as $index => $set) {
            $this->tag->addAttribute(
                sprintf('data-toubiz-standalone-map.map.src-set.%d.width', $index),
                '(int)' . $set['width']
            );
            $this->tag->addAttribute(
                sprintf('data-toubiz-standalone-map.map.src-set.%d.height', $index),
                '(int)' . $set['height']
            );

            if (array_key_exists('query', $set) && $set['query']) {
                $this->tag->addAttribute(
                    sprintf('data-toubiz-standalone-map.map.src-set.%d.query', $index),
                    '(string)' . $set['query']
                );
            }
        }
    }
}
