<?php declare(strict_types=1);

namespace Newland\Toubiz\Map\Neos\Provider\DefaultProviders;

use Neos\ContentRepository\Domain\Model\Node;
use Neos\ContentRepository\Domain\Model\NodeInterface;
use Neos\Flow\Configuration\ConfigurationManager;
use Newland\NeosCommon\Service\NodeService;
use Newland\NeosIcons\IconUrlService;
use Newland\Toubiz\Events\Neos\Service\EventUrlService;
use Newland\Toubiz\Map\Neos\Provider\Contract\Marker;
use Newland\Toubiz\Sync\Neos\Domain\Model\Category;
use Newland\Toubiz\Sync\Neos\Domain\Model\Event;

trait EventFormatting
{
    /** @var NodeService */
    protected $nodeService;

    public function injectNodeService(NodeService $nodeService): void
    {
        $this->nodeService = $nodeService;
    }

    /** @var EventUrlService */
    protected $eventUrlService;

    public function injectEventUrlService(EventUrlService $eventUrlService): void
    {
        $this->eventUrlService = $eventUrlService;
    }

    /** @var IconUrlService */
    protected $iconUrlService;

    public function injectEventIconService(IconUrlService $iconUrlService): void
    {
        $this->iconUrlService = $iconUrlService;
    }

    /** @var array */
    protected $defaultIconSettings;

    public function injectDefaultIconSettings(ConfigurationManager $configurationManager): void
    {
        $this->defaultIconSettings = (array) $configurationManager->getConfiguration(
            ConfigurationManager::CONFIGURATION_TYPE_SETTINGS,
            'Newland.Toubiz.Map.Neos.markers.defaultIcons'
        );
    }

    protected function eventToMarker(
        Event $event,
        Node $eventNode
    ): ?Marker {
        // Event with to geo coordinates should not be selectable, but we filter again to be sure.
        if (!$event->getLocation() || !$event->getLocation()->getLatitude()) {
            return null;
        }

        $marker = new Marker(
            $event->getPersistenceObjectIdentifier(),
            [
                sprintf(
                    '%s/%s',
                    $eventNode->findNodePath(),
                    $event->getPersistenceObjectIdentifier()
                ),
            ],
            (float) $event->getLocation()->getLongitude(),
            (float) $event->getLocation()->getLatitude(),
            $event->getTitle()
        );

        $marker->categoryTitle = $this->getCategoryTitle($event, $eventNode);
        $marker->url = $this->eventUrlService->generateUrlByCurrentNode($event, $eventNode);

        $medium = $event->getMedia()->first();
        if ($medium) {
            $marker->imageUrl = $medium->getPreviewUri();
            $marker->imageAlt = $medium->getTitle();
        }

        $marker->defaultStyle = $this->defaultIconSettings['event'];

        $marker->additionalProperties = [ 'detailApiEndpoints' => (object) [] ];
        return $marker;
    }

    private function getCategoryTitle(Event $event, NodeInterface $eventNode): string
    {
        /** @var Category|null $category */
        $category = $event->getCategories()->first();
        if ($category !== null) {
            return $category->getTitle();
        }

        $parent = $this->nodeService->getClosestParentOfType(
            $eventNode,
            'Newland.Toubiz.Map.Neos:Map.FilterItem'
        );
        if ($parent !== null) {
            return (string) $parent->getProperty('title');
        }

        return '';
    }
}
