<?php declare(strict_types=1);

namespace Newland\Toubiz\Map\Neos\Provider\DefaultProviders;

use Neos\ContentRepository\Domain\Model\Node;
use Neos\ContentRepository\Domain\Model\NodeInterface;
use Neos\Flow\Configuration\ConfigurationManager;
use Newland\NeosCommon\Service\NodeService;
use Newland\NeosIcons\IconUrlService;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\TourAttributes;
use Newland\Toubiz\Events\Neos\Service\EventUrlService;
use Newland\Toubiz\Map\Neos\Provider\Contract\Marker;
use Newland\Toubiz\Poi\Neos\Service\ArticleUrlService;
use Newland\Toubiz\Sync\Neos\Domain\Model\Article;
use Newland\Toubiz\Sync\Neos\Domain\Model\Event;
use Newland\Toubiz\Sync\Neos\Domain\Model\Traits\HasCategoriesInterface;

trait ItemFormatting
{
    /** @var NodeService */
    protected $nodeService;

    public function injectNodeService(NodeService $nodeService): void
    {
        $this->nodeService = $nodeService;
    }

    /** @var ArticleUrlService */
    protected $articleUrlService;

    public function injectArticleUrlService(ArticleUrlService $articleUrlService): void
    {
        $this->articleUrlService = $articleUrlService;
    }

    /** @var EventUrlService */
    protected $eventUrlService;

    public function injectEventUrlService(EventUrlService $eventUrlService): void
    {
        $this->eventUrlService = $eventUrlService;
    }

    /** @var IconUrlService */
    protected $iconUrlService;

    public function injectArticleIconService(IconUrlService $iconUrlService): void
    {
        $this->iconUrlService = $iconUrlService;
    }

    /** @var array */
    protected $defaultIconSettings;

    public function injectDefaultIconSettings(ConfigurationManager $configurationManager): void
    {
        $this->defaultIconSettings = (array) $configurationManager->getConfiguration(
            ConfigurationManager::CONFIGURATION_TYPE_SETTINGS,
            'Newland.Toubiz.Map.Neos.markers.defaultIcons'
        );
    }

    protected function articleToMarker(
        Article $article,
        Node $node,
        bool $asyncTourGeometry = true
    ): ?Marker {
        if (!$article->getHasGeocoordinates()) {
            return null;
        }

        $marker = new Marker(
            $article->getPersistenceObjectIdentifier(),
            [
                sprintf(
                    '%s/%s',
                    $node->findNodePath(),
                    $article->getPersistenceObjectIdentifier()
                ),
            ],
            (float) $article->getLongitude(),
            (float) $article->getLatitude(),
            $article->getName()
        );

        $marker->categoryTitle = $this->getCategoryTitle($article, $node);
        $marker->url = $this->articleUrlService->generateUrlByCurrentNode($article, $node);
        $marker->outdoorActiveTrackingId = $article->getOutdoorActiveId();

        $medium = $article->getMainMedium();
        if ($medium !== null) {
            $marker->imageUrl = $medium->getPreviewUri();
            $marker->imageAlt = $medium->getTitle();
        }

        $marker->defaultStyle = $this->defaultIconSettings['mainType'][$article->getMainType()] ?? [];
        if ($node instanceof NodeInterface) {
            $marker->defaultStyle['iconUrl'] = $this->iconUrlService->buildMarkerIconUrlFromNode(
                $node,
                $marker->defaultStyle
            );
        }

        $detailEndpoints = [];
        if ($article->getIsTour()) {
            $marker->tour = [
                'difficultyRating' => $article->getFirstAttribute(TourAttributes::DIFFICULTY_RATING),
            ];

            if ($asyncTourGeometry) {
                $detailEndpoints['tour.polygon'] = $this->articleUrlService->tourGeometryApiUrl($article, $node);
            } else {
                $marker->tour['polygon'] = $article->getFirstAttribute(TourAttributes::GEOMETRY) ?: [];
            }
        }

        if ($article->getOpeningTimes() !== null) {
            $detailEndpoints['openingTimes'] = $this->articleUrlService->openingTimesApiUrl($article, $node);
        }

        $marker->additionalProperties = [ 'detailApiEndpoints' => (object) $detailEndpoints ];
        return $marker;
    }

    protected function eventToMarker(
        Event $event,
        Node $mapNode
    ): ?Marker {
        if (!$event->getLocation() || !$event->getLocation()->getHasGeocoordinates()) {
            return null;
        }

        $marker = new Marker(
            $event->getPersistenceObjectIdentifier(),
            [
                sprintf(
                    '%s/%s',
                    $mapNode->findNodePath(),
                    $event->getPersistenceObjectIdentifier()
                ),
            ],
            (float) $event->getLocation()->getLongitude(),
            (float) $event->getLocation()->getLatitude(),
            $event->getTitle()
        );

        $marker->categoryTitle = $this->getCategoryTitle($event, $mapNode);
        $marker->url = $this->eventUrlService->generateUrlByCurrentNode($event, $mapNode);

        $medium = $event->getMedia()->first();
        if ($medium) {
            $marker->imageUrl = $medium->getPreviewUri();
            $marker->imageAlt = $medium->getTitle();
        }

        $marker->defaultStyle = $this->defaultIconSettings['event'] ?? [];
        if ($mapNode instanceof NodeInterface) {
            $marker->defaultStyle['iconUrl'] = $this->iconUrlService->buildMarkerIconUrlFromNode(
                $mapNode,
                $marker->defaultStyle
            );
        }

        $marker->additionalProperties = [ 'detailApiEndpoints' => (object) [] ];
        return $marker;
    }

    private function getCategoryTitle(HasCategoriesInterface $item, NodeInterface $node): string
    {
        $category = $item->getMainCategory();
        if ($category !== null) {
            return $category->getTitle();
        }

        $parent = $this->nodeService->getClosestParentOfType(
            $node,
            'Newland.Toubiz.Map.Neos:Map.FilterItem'
        );
        if ($parent !== null) {
            return (string) $parent->getProperty('title');
        }

        return '';
    }
}
