<?php declare(strict_types=1);

namespace Newland\Toubiz\Map\Neos\Provider;

use Neos\ContentRepository\Domain\Factory\NodeFactory;
use Neos\ContentRepository\Domain\Model\Node;
use Neos\ContentRepository\Domain\Model\NodeData;
use Neos\ContentRepository\Domain\Model\Workspace;
use Neos\ContentRepository\Domain\Projection\Content\NodeInterface;
use Neos\ContentRepository\Domain\Projection\Content\TraversableNodeInterface;
use Neos\ContentRepository\Domain\Repository\NodeDataRepository;
use Neos\Neos\Service\PublishingService;
use Neos\Neos\Ui\ContentRepository\Service\NodeService;
use Neos\ContentRepository\Exception\NodeException;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Persistence\PersistenceManagerInterface;
use Neos\Neos\Domain\Service\ContentContextFactory;
use Neos\Neos\Ui\Controller\BackendServiceController;
use Newland\Toubiz\Map\Neos\Provider\Contract\ProviderContext;
use Newland\Toubiz\Map\Neos\Provider\DefaultProviders\NodeBasedFilterItems;

/**
 * Slot that forces regeneration of the map serialization cache every time the nodes are changed.
 *
 * @Flow\Aspect()
 * @Flow\Scope("singleton")
 */
class NodeSlot
{
    /**
     * @var MapDataProviderService
     * @Flow\Inject()
     */
    protected $mapDataProvider;

    /**
     * @var NodeDataRepository
     * @Flow\Inject()
     */
    protected $nodeDataRepository;

    /**
     * @var NodeFactory
     * @Flow\Inject()
     */
    protected $nodeFactory;

    /**
     * @var ContentContextFactory
     * @Flow\Inject()
     */
    protected $contentContextFactory;

    /**
     * @var PersistenceManagerInterface
     * @Flow\Inject()
     */
    protected $persistenceManager;

    /**
     * @var NodeService
     * @Flow\Inject()
     */
    protected $nodeService;

    /** @var string[] */
    protected $publishedNodes = [];

    public function onNodePropertyChanged(TraversableNodeInterface $node): void
    {
        $this->persistenceManager->persistAll();
        if (!$node->getNodeType()->isOfType(MapDataProviderService::TYPE_MARKERS_WILDCARD)) {
            return;
        }
        $this->mapDataProvider->clearCache();
    }

    public function onNodePathChanged(TraversableNodeInterface $node): void
    {
        $this->persistenceManager->persistAll();
        if (!$node->getNodeType()->isOfType(NodeBasedFilterItems::FILTER_ITEM)) {
            return;
        }
        $this->mapDataProvider->clearCache();
    }

    /**
     * Listener for the `onNodePublished` event on the PublishingService.
     * Saves information about all published map nodes for later usage to re-prime caches.
     *
     * @see PublishingService::emitNodePublished()
     * @see self::afterPublishing()
     */
    public function onNodePublished(NodeInterface $node, Workspace $targetWorkspace): void
    {
        $mapNode = $this->mapDataProvider->getMapNode($node);
        $nodeInTargetWorkspace = $this->fetchNodeInGivenWorkspace($mapNode, $targetWorkspace);

        if (!($nodeInTargetWorkspace instanceof \Neos\ContentRepository\Domain\Model\NodeInterface)) {
            return;
        }

        $path = $nodeInTargetWorkspace->getContextPath();
        if (!\in_array($path, $this->publishedNodes, true)) {
            $this->publishedNodes[] = $path;
        }
    }

    /**
     * Primes caches for nodes that have been published in this request.
     * This is done after the API request has finished instead of in the event listener in
     * order to prevent persistence issues where data is not yet in a consistent state.
     *
     * @see BackendServiceController::publishAction()
     * @see self::onNodePublished
     * @Flow\After("method(Neos\Neos\Ui\Controller\BackendServiceController->publishAction())")
     */
    public function afterPublishing(): void
    {
        $this->mapDataProvider->clearCache();
        foreach ($this->publishedNodes as $contextPath) {
            $node = $this->nodeService->getNodeFromContextPath($contextPath);
            if ($node instanceof Node) {
                $context = new ProviderContext($node, null);
                $this->mapDataProvider->getMarkers($context);
                $this->mapDataProvider->getFilterItems($context);
            }
        }
    }

    private function fetchNodeInGivenWorkspace(
        ?TraversableNodeInterface $node,
        Workspace $workspace
    ): ?TraversableNodeInterface {

        if ($node === null) {
            return null;
        }

        /** @var NodeData|null $data */
        $data = $this->nodeDataRepository->findOneByPath((string) $node->findNodePath(), $workspace);
        if ($data === null) {
            return null;
        }

        $nodeInTargetWorkspace = $this->nodeFactory->createFromNodeData(
            $data,
            $this->contentContextFactory->create(
                [
                    'workspaceName' => $data->getWorkspace()->getName(),
                    'invisibleContentShown' => false,
                    'inaccessibleContentShown' => false,
                    'removedContentShown' => false,
                    'dimensions' => $data->getDimensionValues(),
                ]
            )
        );

        if (!($nodeInTargetWorkspace instanceof TraversableNodeInterface)) {
            return null;
        }

        return $nodeInTargetWorkspace;
    }
}
