<?php declare(strict_types=1);
namespace Newland\Toubiz\Map\Neos\Serialization\Markers;

/*
 * This file is part of the "toubiz-map-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\ContentRepository\Domain\Model\NodeInterface;
use Neos\ContentRepository\Domain\Projection\Content\TraversableNodeInterface;
use Neos\Flow\Configuration\ConfigurationManager;
use Newland\NeosCommon\Service\NodeService;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\TourAttributes;
use Newland\NeosIcons\IconUrlService;
use Newland\Toubiz\Poi\Neos\Service\ArticleUrlService;
use Newland\Toubiz\Sync\Neos\Domain\Model\Article;

trait ArticleFormatting
{
    /** @var NodeService */
    protected $nodeService;

    public function injectNodeService(NodeService $nodeService): void
    {
        $this->nodeService = $nodeService;
    }

    /** @var ArticleUrlService */
    protected $articleUrlService;

    public function injectArticleUrlService(ArticleUrlService $articleUrlService): void
    {
        $this->articleUrlService = $articleUrlService;
    }

    /** @var IconUrlService */
    protected $iconUrlService;

    public function injectArticleIconService(IconUrlService $iconUrlService): void
    {
        $this->iconUrlService = $iconUrlService;
    }

    /** @var array */
    protected $defaultIconSettings;

    public function injectDefaultIconSettings(ConfigurationManager $configurationManager): void
    {
        $this->defaultIconSettings = (array) $configurationManager->getConfiguration(
            ConfigurationManager::CONFIGURATION_TYPE_SETTINGS,
            'Newland.Toubiz.Map.Neos.markers.defaultIcons'
        );
    }

    protected function articleToMarker(
        Article $article,
        TraversableNodeInterface $articleNode,
        array $additionalProperties = []
    ): array {
        $medium = $article->getMainMedium();
        $image = null;
        if ($medium !== null) {
            $image = [
                'url' => $medium->getSourceUri(),
                'alt' => $medium->getTitle(),
            ];
        }

        $detailApiEndpoints = new \stdClass();
        $url = null;
        if ($articleNode instanceof NodeInterface) {
            $url = $this->articleUrlService->generateUrlByCurrentNode($article, $articleNode);
            if ($article->getOpeningTimes() !== null) {
                $detailApiEndpoints->openingTimes =
                    $this->articleUrlService->openingTimesApiUrl($article, $articleNode);
            }
        }

        $style = $this->defaultIconSettings['mainType'][$article->getMainType()] ?? [];
        if ($articleNode instanceof NodeInterface) {
            $style['iconUrl'] = $this->iconUrlService->buildMarkerIconUrlFromNode($articleNode, $style);
        }

        $formatted = [
            'id' => $article->getPersistenceObjectIdentifier(),
            'paths' => [ sprintf(
                '%s/%s',
                $articleNode->findNodePath(),
                $article->getPersistenceObjectIdentifier()
            ) ],
            'title' => $article->getName(),
            'categoryTitle' => $this->getCategoryTitle($article, $articleNode),
            'url' => $url,
            'image' => $image,
            'coordinates' => [
                'longitude' => $article->getLongitude(),
                'latitude' => $article->getLatitude(),
            ],
            'defaultStyle' => $style,
            'tour' => $article->getIsTour() ? [
                'polygon' => $article->getFirstAttribute(TourAttributes::GEOMETRY) ?: [],
                'difficultyRating' => $article->getFirstAttribute(TourAttributes::DIFFICULTY_RATING),
            ] : null,
            'outdoorActiveTrackingId' => $article->getOutdoorActiveId(),
            'detailApiEndpoints' => $detailApiEndpoints,
        ];

        return (array) array_replace_recursive($additionalProperties, $formatted);
    }

    private function getCategoryTitle(Article $article, TraversableNodeInterface $articleNode): string
    {
        $category = $article->getMainCategory();
        if ($category !== null) {
            return $category->getTitle();
        }

        if ($articleNode instanceof NodeInterface) {
            $parent = $this->nodeService->getClosestParentOfType(
                $articleNode,
                'Newland.Toubiz.Map.Neos:Map.FilterItem'
            );
            if ($parent !== null) {
                return (string) $parent->getProperty('title');
            }
        }

        return '';
    }
}
