<?php declare(strict_types=1);

namespace Newland\Toubiz\Map\Neos\Provider\DefaultProviders;

use Neos\ContentRepository\Domain\Model\Node;
use Neos\ContentRepository\Domain\Model\NodeInterface;
use Neos\Flow\Configuration\Exception\InvalidConfigurationException;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Map\Neos\Provider\Contract\MarkerProvider;
use Newland\Toubiz\Map\Neos\Provider\MapDataProvider;
use Newland\Toubiz\Poi\Neos\Filter\ArticleFilterFactory;
use Newland\Toubiz\Sync\Neos\Domain\Model\Article;
use Neos\Flow\Annotations as Flow;

/**
 * @MapDataProvider()
 */
class FilteredArticles extends SubnodeBasedMarkerProvider
{
    use ArticleFormatting;
    protected $nodeType = 'Newland.Toubiz.Map.Neos:Map.Markers.FilteredArticles';

    /**
     * @var int
     * @Flow\InjectConfiguration(package="Newland.Toubiz.Map.Neos", path="filteredArticles.limit")
     */
    protected $limit;


    public function markersForNode(Node $node): array
    {

        if (!$node->hasProperty('articleType')) {
            return [];
        }

        $mainType = (int) $node->getProperty('articleType');
        if (!\in_array($mainType, ArticleConstants::ALL_TYPES, true)) {
            throw new InvalidConfigurationException(
                sprintf(
                    '"%s" is not a valid article type. Must be one of [%s]',
                    $node->getProperty('articleType'),
                    implode(', ', ArticleConstants::ALL_TYPES)
                )
            );
        }

        if (!($node instanceof NodeInterface)) {
            return [];
        }

        /** @var Article[] $articles */
        $articles = (new ArticleFilterFactory($node))
            ->createFilterForArticleType($mainType, (array) $node->getProperties())
            ->setPageSize($this->limit)
            ->getArticleQuery()
            ->getQuery()
            ->execute();

        $alwaysDisplayPolygon = (bool) $node->getProperty('alwaysDisplayPolygon');
        $color = $node->getProperty('tourColor');
        $markers = [];
        foreach ($articles as $article) {
            $marker = $this->articleToMarker($article, $node, !$alwaysDisplayPolygon);
            if (!$marker) {
                continue;
            }

            if ($marker->tour) {
                $marker->tour['alwaysDisplayPolygon'] = $alwaysDisplayPolygon;
                $marker->tour['color'] = $color;
            }

            $markers[$article->getPersistenceObjectIdentifier()] = $marker;
        }

        return $markers;
    }
}
