<?php
namespace Newland\Toubiz\Map\Neos\Controller;

/*
 * This file is part of the "toubiz-map-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\ContentRepository\Domain\Model\Node;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Mvc\Controller\ActionController;
use Newland\Toubiz\Map\Neos\Utility\CategoryMappingUtility;
use Newland\Toubiz\Sync\Neos\Domain\Filter\ArticleFilter;
use Newland\Toubiz\Sync\Neos\Domain\Repository\ArticleRepository;
use Newland\Toubiz\Map\Neos\Service\ArticleCacheService;

/**
 * Map controller.
 *
 * @Flow\Scope("singleton")
 */
class MapController extends ActionController
{
    /**
     * @var ArticleRepository
     * @Flow\Inject
     */
    protected $articleRepository;

    /**
     * @Flow\InjectConfiguration
     * @var array
     */
    protected $configuration;

    /**
     * @var array
     */
    protected $properties = [];

    /**
     * @Flow\Inject
     * @var ArticleCacheService
     */
    protected $articleCacheService;

    /**
     * @Flow\InjectConfiguration(package="Newland.Toubiz.Sync.Neos.services.Outdooractive/Api")
     * @var array
     */
    protected $outdooractiveCredentials;

    /**
     * @var Node
     */
    protected $node;

    public static function createArticleFilter(array $properties, array $configuration): ArticleFilter
    {
        $articleFilter = new ArticleFilter;
        if (!empty($properties['categories'])) {
            $categoryIds = CategoryMappingUtility::getMappedIdsFromConfiguration(
                $configuration['categories'],
                $properties['categories']
            );

            $articleFilter->setCategoriesIdentifierField('originalId');
            $articleFilter->setCategories($categoryIds);
        }
        $articleFilter->setExcludeUnsafeCoordinates(true);
        return $articleFilter;
    }


    /**
     * Action initializer.
     *
     * @return void
     */
    public function initializeAction()
    {
        $this->documentNode = $this->request->getInternalArgument('__documentNode');
        $this->node = $this->request->getInternalArgument('__node');
        if ($this->node && $this->node->getProperties()) {
            $this->properties = $this->node->getProperties();
        }
    }

    /**
     * Show action.
     *
     * Displays the map.
     *
     * @return void
     */
    public function showAction()
    {
        $isInBackend = $this->node->getContext()->isInBackend();

        $articleJson = '[]';
        if (!$isInBackend) {
            $filter = static::createArticleFilter($this->properties, $this->configuration);
            $articleJson = $this->articleCacheService->cacheQueryResultsAsJson(
                $this->articleRepository->queryForFilter($filter),
                $this->controllerContext
            );
        }

        $this->view->assignMultiple([
            'configuration' => $this->configuration,
            'properties' => $this->properties,
            'node' => $this->node,
            'nodes' => $this->request->getInternalArgument('__mapNodes'),
            'categories' => $this->getCategories(),
            'articleJson' => $articleJson,
            'outdooractiveCredentials' => $this->outdooractiveCredentials,
        ]);
    }


    /**
     * Returns configured categories.
     *
     * Optionally filters configured categories based on
     * the given sub-configuration (i.e. from a node).
     *
     * @return array
     */
    protected function getCategories()
    {
        if (!array_key_exists('categories', $this->configuration)) {
            // No search without a haystack.
            return [];
        }

        if (array_key_exists('categories', $this->properties)
            && !empty($this->properties['categories'])
        ) {
            return CategoryMappingUtility::getCategoriesFromConfiguration(
                $this->configuration['categories'],
                $this->properties['categories']
            );
        } else {
            // No filtering without needles.
            return $this->configuration['categories'];
        }
    }

}
