<?php
namespace Newland\Toubiz\Map\Neos\Service;

use Doctrine\ORM\QueryBuilder;
use Neos\Cache\Frontend\FrontendInterface;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Cache\CacheManager;
use Neos\Flow\Mvc\Controller\ControllerContext;
use Newland\Toubiz\Map\Neos\Exception\PoiPackageRequiredException;
use Newland\Toubiz\Poi\Neos\Service\ArticleUrlService;
use Newland\Toubiz\Map\Neos\DependencyInjection\SoftInject;

/**
 * @Flow\Scope("singleton")
 */
class ArticleCacheService
{
    const CACHE_NAME = 'Newland_Toubiz_Map_Neos-Data';

    /**
     * @Flow\Inject
     * @var CacheManager
     */
    protected $cacheManager;

    /**
     * @SoftInject(errorMessage="Package Newland.Toubiz.Poi.Neos is required for this action.")
     * @var ArticleUrlService
     */
    protected $articleUrl;

    public function cacheQueryResultsAsJson(QueryBuilder $query, ControllerContext $context): string
    {
        return $this->generateWithoutCachingIfNotPublicContext($query, $context) ?:
            $this->fetchFromCache($query) ?:
            $this->generateAndStoreInCache($query, $context);
    }

    public function generateAndStoreInCache(QueryBuilder $query, ControllerContext $context): string
    {
        $cache = $this->cacheManager->getCache(static::CACHE_NAME);
        $json = $this->generate($query->getQuery()->execute(), $context);
        $cache->set($this->cacheKey($query), $json);
        return $json;
    }

    private function generateWithoutCachingIfNotPublicContext(QueryBuilder $query, ControllerContext $context)
    {
        if (!$context->getRequest()->getInternalArgument('__node')->getContext()->getWorkspace()->isPublicWorkspace()) {
            return $this->generate($query->getQuery()->execute(), $context);
        }
        return null;
    }

    private function fetchFromCache(QueryBuilder $query)
    {
        $cache = $this->cacheManager->getCache(static::CACHE_NAME);
        $cacheKey = $this->cacheKey($query);

        if ($cache->has($cacheKey)) {
            return $cache->get($cacheKey);
        }
        return null;
    }


    private function cacheKey(QueryBuilder $query): string
    {
        return md5($query->getDQL());
    }

    private function generate(array $articles, ControllerContext $context): string
    {
        $cached = [];
        foreach ($articles as $article) {
            $cached[] = array_merge(
                $article->jsonSerialize(),
                [ 'url' => $this->articleUrl->generateUrl($article, $context) ]
            );
        }

        return json_encode($cached);
    }

}