<?php
namespace Newland\Toubiz\Map\Neos\Domain\Repository;

use Doctrine\Common\Persistence\ObjectManager;
use Doctrine\ORM\QueryBuilder;
use Neos\ContentRepository\Domain\Factory\NodeFactory;
use Neos\ContentRepository\Domain\Model\Node;
use Neos\ContentRepository\Domain\Model\NodeData;
use Neos\Flow\Annotations as Flow;
use Neos\Neos\Domain\Model\Site;
use Neos\Neos\Domain\Repository\SiteRepository;
use Neos\Neos\Domain\Service\ContentContext;
use Neos\Neos\Domain\Service\ContentContextFactory;

/**
 * @Flow\Scope("singleton")
 */
class NodeRepository
{

    /**
     * @Flow\Inject
     * @var ObjectManager
     */
    protected $entityManager;

    /**
     * @Flow\Inject
     * @var NodeFactory
     */
    protected $nodeFactory;

    /**
     * @Flow\Inject
     * @var ContentContextFactory
     */
    protected $contextFactory;

    /**
     * @Flow\Inject
     * @var SiteRepository
     */
    protected $siteRepository;


    /**
     * Fetches all nodes of the given type.
     * If the second argument is set to `true` then the site the node belongs to is manually
     * resolved by traversing the parents instead of relying on the built-in resolution of Neos.
     *
     * This is useful because Neos caches the first domain it resolves and assumes that a process
     * only lives as long as a request to a single domain. This assumption is false for CLI jobs
     * that process multiple nodes of different domains.
     *
     * @param string $nodeType
     * @param bool $manualSiteResolution
     * @return Node[]
     */
    public function findNodesByNodeType(string $nodeType, bool $manualSiteResolution = false): array
    {
        /** @var QueryBuilder $queryBuilder */
        $queryBuilder = $this->entityManager->createQueryBuilder();
        $queryBuilder->select('n')->from(NodeData::class, 'n')
            ->where('n.nodeType = :nodeType')
            ->setParameter('nodeType', $nodeType);
        $nodeData = $queryBuilder->getQuery()->execute();

        return array_map(
            function(NodeData $data) use ($manualSiteResolution) {
                return $this->nodeFactory->createFromNodeData(
                    $data,
                    $this->createContentContext($data, $manualSiteResolution)
                );
            },
            $nodeData
        );
    }

    private function createContentContext(NodeData $data, bool $manualSiteResolution): ContentContext
    {
        return $this->contextFactory->create([
             'workspaceName' => $data->getWorkspace()->getName(),
             'dimensions' => $data->getDimensionValues(),
             'currentSite' => $manualSiteResolution ? $this->siteForNodeData($data) : null,
             'invisibleContentShown' => true,
             'inaccessibleContentShown' => true,
             'removedContentShown' => true,
         ]);
    }

    private function siteForNodeData(NodeData $data): Site
    {
        do {
            $site = $this->siteRepository->findOneByNodeName($data->getName());
            if ($site) {
                return $site;
            }
        } while ($data = $data->getParent());

        return $this->siteRepository->findDefault();
    }

}