<?php
namespace Newland\Toubiz\Map\Neos\Command;

/*
 * This file is part of the "toubiz-map-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Doctrine\Common\Persistence\ObjectManager;
use Doctrine\ORM\QueryBuilder;
use Neos\ContentRepository\Domain\Factory\NodeFactory;
use Neos\ContentRepository\Domain\Model\Node;
use Neos\ContentRepository\Domain\Model\NodeData;
use Neos\Flow\Cli\CommandController;
use Neos\Flow\Http\Request;
use Neos\Flow\Http\Response;
use Neos\Flow\Http\Uri;
use Neos\Flow\Mvc\ActionRequest;
use Neos\Flow\Mvc\Controller\Arguments;
use Neos\Flow\Mvc\Controller\ControllerContext;
use Neos\Flow\Mvc\Routing\UriBuilder;
use Neos\Neos\Domain\Model\Domain;
use Newland\Toubiz\Map\Neos\Controller\MapController;
use Newland\Toubiz\Map\Neos\Service\ArticleCacheService;
use Newland\Toubiz\Sync\Neos\Domain\Repository\ArticleRepository;

/**
 * Map command controller.
 *
 * Provides commands to manipulate map data.
 *
 * @Flow\Scope("singleton")
 */
class MapCommandController extends CommandController
{
    /**
     * @var ArticleRepository
     * @Flow\Inject
     */
    protected $articleRepository;

    /**
     * @Flow\Inject
     * @var ObjectManager
     */
    protected $entityManager;

    /**
     * @Flow\InjectConfiguration(package="Newland.Toubiz.Map.Neos")
     * @var array
     */
    protected $configuration;

    /**
     * @Flow\Inject
     * @var NodeFactory
     */
    protected $nodeFactory;

    /**
     * @Flow\Inject
     * @var ArticleCacheService
     */
    protected $articleCache;

    /**
     * Command to build an article cache to feed the map.
     *
     * As the map is based on a lot of marker data which requires a lot of
     * computing resources when fetching on-demand, static data is
     * built to assign to the view. This reduces server load.
     *
     * @return void
     */
    public function buildArticleCacheCommand()
    {
        foreach ($this->getNodes() as $node) {
            echo 'Warming cache for ' . $node->getName() . "\n";

            $query = $this->articleRepository->queryForFilter(
                MapController::createArticleFilter($node->getProperties(), $this->configuration)
            );

            $this->articleCache->generateAndStoreInCache($query, $this->initializeFakeControllerContext($node));
        }
    }

    /** @return Node[] */
    private function getNodes(): array
    {
        /** @var QueryBuilder $queryBuilder */
        $queryBuilder = $this->entityManager->createQueryBuilder();
        $queryBuilder->select('n')->from(NodeData::class, 'n')
            ->where('n.nodeType = :nodeType')
            ->setParameter('nodeType', 'Newland.Toubiz.Map.Neos:Map');
        $nodeData = $queryBuilder->getQuery()->execute();

        return array_map(
            function(NodeData $data) {
                $context = $this->nodeFactory->createContextMatchingNodeData($data);
                return $this->nodeFactory->createFromNodeData($data, $context);
            },
            $nodeData
        );
    }

    private function initializeFakeControllerContext(Node $node): ControllerContext
    {
        /** @var Domain $domain */
        $domain = $node->getContext()->getCurrentSite()->getPrimaryDomain();

        $httpRequest = new Request([], [], [], [ 'FLOW_REWRITEURLS' => 1, 'SCRIPT_NAME' => '/' ]);
        $httpRequest->setContent(null);
        $httpRequest->setBaseUri(new Uri($domain->__toString() . '/'));

        $request = new ActionRequest($httpRequest);
        $request->setArgument('__node', $node);

        $uriBuilder = new UriBuilder();
        $uriBuilder->setRequest($request);

        return new ControllerContext(
            $uriBuilder->getRequest(),
            new Response(),
            new Arguments([]),
            $uriBuilder
        );
    }

}
