<?php

namespace Newland\Toubiz\Map\Neos\Serialization;

use Neos\ContentRepository\Domain\Model\Node;
use Neos\ContentRepository\Domain\Model\NodeInterface;
use Neos\ContentRepository\Domain\NodeType\NodeTypeConstraints;
use Neos\ContentRepository\Domain\NodeType\NodeTypeName;
use Neos\ContentRepository\Domain\Projection\Content\TraversableNodeInterface;
use Neos\Eel\FlowQuery\FlowQuery;
use Neos\Eel\ProtectedContextAwareInterface;
use Neos\Flow\Configuration\Exception\InvalidConfigurationException;
use Neos\Flow\ObjectManagement\ObjectManager;
use Neos\Neos\Domain\Service\ContentContext;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\TourAttributes;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\NeosIcons\IconUrlService;
use Newland\Toubiz\Map\Neos\Serialization\Markers\FilteredArticles;
use Newland\Toubiz\Map\Neos\Serialization\Markers\Links;
use Newland\Toubiz\Map\Neos\Serialization\Markers\NodeSerializer;
use Newland\Toubiz\Map\Neos\Serialization\Markers\SelectedArticles;
use Newland\Toubiz\Map\Neos\Serialization\Markers\SelectedPages;
use Newland\Toubiz\Poi\Neos\Filter\ArticleFilterFactory;
use Neos\Flow\Annotations as Flow;
use Newland\Toubiz\Poi\Neos\Service\ArticleUrlService;
use Newland\Toubiz\Sync\Neos\Domain\Model\Article;

/**
 * This class contains the raw serialization logic.
 * If you want to serialize filter items and markers for frontend usage you may
 * want to use `SerializationService` instead as it also handles caching of results.
 *
 * @see SerializationService
 * @internal
 * @Flow\Scope("singleton")
 */
class MapSerializer
{
    public const TYPE_MAP = 'Newland.Toubiz.Map.Neos:Map';
    public const TYPE_MARKERS_WILDCARD = 'Newland.Toubiz.Map.Neos:Map.Markers';
    public const TYPE_FILTER_ITEM = 'Newland.Toubiz.Map.Neos:Map.FilterItem';

    public const ALL_TYPES = [
        self::TYPE_MAP,
        self::TYPE_MARKERS_WILDCARD,
        self::TYPE_FILTER_ITEM,
    ];

    /**
     * @var ObjectManager
     * @Flow\Inject()
     */
    protected $objectManager;

    /** @var string[] */
    protected $markerFormatterClassNames = [
        FilteredArticles::class,
        SelectedArticles::class,
        SelectedPages::class,
        Links::class,
    ];

    /**
     * @var IconUrlService
     * @Flow\Inject()
     */
    protected $iconUrlService;

    /** @var NodeSerializer[] */
    protected $markerFormatterInstances;

    public function injectMarkerFormatterInstances(ObjectManager $objectManager): void
    {
        $this->markerFormatterInstances = array_map(
            function (string $className) use ($objectManager) {
                return $objectManager->get($className);
            },
            $this->markerFormatterClassNames
        );
    }

    public function serializeFilterItems(TraversableNodeInterface $node, array $style = []): array
    {
        $filterItems = [];
        $openByDefaultAlreadySet = false;

        foreach ($this->childrenOfContentCollections($node, [ static::TYPE_FILTER_ITEM ]) as $child) {
            $style['icon'] = $child->getProperty('icon') ?? $style['icon'] ?? null;
            $style['backgroundColor'] = $child->getProperty('backgroundColor') ?? $style['backgroundColor'] ?? null;
            $style['color'] = $child->getProperty('color') ?? $style['color'] ?? null;

            if ($node instanceof NodeInterface) {
                $style['iconUrl'] = $this->iconUrlService->buildMarkerIconUrlFromNode($node, $style);
            }

            $openByDefault = false;
            if (!$openByDefaultAlreadySet && (bool) $child->getProperty('openByDefault')) {
                $openByDefault = true;
                $openByDefaultAlreadySet = true;
            }

            $filterItems[$child->getIdentifier()] = [
                'path' => (string) $child->findNodePath(),
                'title' => $child->getProperty('title'),
                'style' => $style,
                'overwriteMarkerStyles' => !$child->getProperty('useDefaultStyles'),
                'openByDefault' => $openByDefault,
                'children' => $this->serializeFilterItems($child, $style),
            ];
        }
        return $filterItems;
    }

    public function serializeMarkers(TraversableNodeInterface $node): array
    {
        $markers = [ ];

        $append = function (string $key, array $item) use (&$markers) {
            if (array_key_exists($key, $markers)) {
                // First appended item wins, only update paths
                $markers[$key]['paths'] = array_merge($markers[$key]['paths'], $item['paths']);
            } else {
                $markers[$key] = $item;
            }
        };

        foreach ($this->findMarkerNodes($node) as $articleNode) {
            foreach ($this->callMarkerFormatter($articleNode) as $key => $formatted) {
                $append($key, $formatted);
            }
        }

        if ($node->getNodeType()->isOfType(static::TYPE_MARKERS_WILDCARD)) {
            foreach ($this->callMarkerFormatter($node) as $key => $formatted) {
                $append($key, $formatted);
            }
        }

        return $markers;
    }

    private function callMarkerFormatter(TraversableNodeInterface $node): array
    {
        foreach ($this->markerFormatterInstances as $formatter) {
            if ($formatter->canSerialize($node)) {
                return $formatter->serialize($node);
            }
        }

        throw new InvalidNodeException(sprintf(
            'Could not serialize node %s of type %s',
            (string) $node->getNodeName(),
            $node->getNodeType()
        ));
    }


    /**
     * @param string[] $nodeTypes
     */
    private function childrenOfContentCollections(TraversableNodeInterface $node, array $nodeTypes): \Generator
    {
        $constraints = new NodeTypeConstraints(
            false,
            [ NodeTypeName::fromString('Neos.Neos:ContentCollection') ]
        );
        $types = array_map(
            function (string $type) {
                return NodeTypeName::fromString($type);
            },
            $nodeTypes
        );

        foreach ($node->findChildNodes($constraints) as $contentCollection) {
            foreach ($contentCollection->findChildNodes(new NodeTypeConstraints(true, $types)) as $childNode) {
                if (!$childNode->isHidden()) {
                    yield $childNode;
                }
            }
        }
    }

    private function findMarkerNodes(TraversableNodeInterface $node): \Generator
    {
        if ($node instanceof NodeInterface && $node->isHidden()) {
            return;
        }

        yield from $this->childrenOfContentCollections($node, [ self::TYPE_MARKERS_WILDCARD ]);

        foreach ($this->childrenOfContentCollections($node, [ self::TYPE_FILTER_ITEM ]) as $childFilterItem) {
            yield from $this->findMarkerNodes($childFilterItem);
        }
    }
}
