<?php declare(strict_types=1);

namespace Newland\Toubiz\Map\Neos\Serialization;

use Neos\Cache\Frontend\FrontendInterface as CacheFrontendInterface;
use Neos\ContentRepository\Domain\Projection\Content\TraversableNodeInterface;
use Neos\Eel\FlowQuery\FlowQuery;
use Neos\Eel\ProtectedContextAwareInterface;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Cache\CacheManager;

/**
 * @Flow\Scope("singleton")
 */
class SerializationService implements ProtectedContextAwareInterface
{
    const CACHE_NAME = 'Newland_Toubiz_Map_Neos-Data';
    const CACHE_KEY_MARKER = 'markers__{identifier}';
    const CACHE_KEY_ITEMS = 'filter-items__{identifier}';
    const SERIALIZATION_TAG = 'newland-toubiz-map-serialization';

    /**
     * @var MapSerializer
     * @Flow\Inject()
     */
    protected $mapSerializer;

    /** @var CacheFrontendInterface */
    protected $cache;
    public function injectCache(CacheManager $cacheManager): void
    {
        $this->cache = $cacheManager->getCache(static::CACHE_NAME);
    }

    public function serializeAndCacheMarkers(TraversableNodeInterface $node): array
    {
        $key = $this->cacheKey(static::CACHE_KEY_MARKER, $node);
        if ($this->cache->has($key)) {
            return $this->cache->get($key);
        }

        $result = $this->mapSerializer->serializeMarkers($node);
        $this->cache->set($key, $result, [ static::SERIALIZATION_TAG ]);
        return $result;
    }

    public function forceRegenerateMarkerCache(TraversableNodeInterface $node): void
    {
        $key = $this->cacheKey(static::CACHE_KEY_MARKER, $node);
        $this->cache->remove($key);
        $this->serializeAndCacheMarkers($node);
    }

    public function serializeAndCacheFilterItems(TraversableNodeInterface $node): array
    {
        $key = $this->cacheKey(static::CACHE_KEY_ITEMS, $node);
        if ($this->cache->has($key)) {
            return $this->cache->get($key);
        }

        $result = $this->mapSerializer->serializeFilterItems($node);
        $this->cache->set($key, $result, [ static::SERIALIZATION_TAG ]);
        return $result;
    }

    public function forceRegenerateFilterItemCache(TraversableNodeInterface $node): void
    {
        $key = $this->cacheKey(static::CACHE_KEY_ITEMS, $node);
        $this->cache->remove($key);
        $this->serializeAndCacheFilterItems($node);
    }

    public function clearAllCachedSerialization(): void
    {
        $this->cache->flushByTag(static::SERIALIZATION_TAG);
    }

    public function forceRegenerateAllCacheEntries(TraversableNodeInterface $node): void
    {
        $this->forceRegenerateFilterItemCache($node);
        $this->forceRegenerateMarkerCache($node);
    }

    public function forceRegenerateAllCacheEntriesOfSubNodes(TraversableNodeInterface $parentNode, callable $cb): void
    {
        $query = array_map(
            function (string $type) {
                return sprintf('[instanceof %s]', $type);
            },
            MapSerializer::ALL_TYPES
        );

        $results = (new FlowQuery([ $parentNode ]))->find(implode(',', $query));
        $total = \count($results);
        $cb(0, $total);

        foreach ($results as $index => $node) {
            $this->forceRegenerateAllCacheEntries($node);
            $cb($index + 1, $total);
        }
    }

    private function cacheKey(string $template, TraversableNodeInterface $node): string
    {
        return str_replace(
            '{identifier}',
            md5($node->getCacheEntryIdentifier()),
            $template
        );
    }

    public function allowsCallOfMethod($methodName)
    {
        return strpos($methodName, 'serializeAndCache') === 0;
    }
}
