<?php declare(strict_types=1);

namespace Newland\Toubiz\Map\Neos\Tests\Unit\Serialization\Markers;

use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Map\Neos\Serialization\Markers\FilteredArticles;
use Newland\Toubiz\Sync\Neos\Tests\Factory\ArticleFactory;
use Newland\Toubiz\Sync\Neos\Tests\Factory\AttributeFactory;
use Newland\Toubiz\Sync\Neos\Tests\Factory\CategoryFactory;

class FilteredArticlesTest extends MarkerSerializerTestCase
{
    /** @var FilteredArticles */
    protected $subject;

    /** @var ArticleFactory */
    protected $articleFactory;

    /** @var AttributeFactory */
    protected $attributeFactory;

    /** @var CategoryFactory */
    protected $categoryFactory;

    public function setUp(): void
    {
        parent::setUp();

        $this->node = $this->initializeNode(
            '/sites/foo/main/map/children/filter-item/foo-child',
            'live',
            $this->site,
            [  ],
            null,
            'Newland.Toubiz.Map.Neos:Map.Markers.FilteredArticles'
        );

        $this->subject = $this->objectManager->get(FilteredArticles::class);
        $this->articleFactory = new ArticleFactory($this->objectManager);
        $this->attributeFactory = new AttributeFactory($this->objectManager);
        $this->categoryFactory = new CategoryFactory($this->objectManager);
    }


    public function testAllowsFilteringArticlesByType(): void
    {
        $attraction = $this->articleFactory->create([ 'mainType' => ArticleConstants::TYPE_ATTRACTION ]);
        $tour = $this->articleFactory->create([ 'mainType' => ArticleConstants::TYPE_TOUR ]);

        $this->node->setProperty('articleType', ArticleConstants::TYPE_ATTRACTION);
        $this->persistNode($this->node);

        $result = $this->subject->serialize($this->node);

        $this->assertCount(1, $result);
        $this->assertEquals($attraction->getName(), array_values($result)[0]['title']);
        $this->assertNotEquals($tour->getName(), array_values($result)[0]['title']);
    }

    public function testAllowsFilteringArticlesByTags(): void
    {
        $this->articleFactory->create(
            [
                'name' => 'foo',
                'mainType' => ArticleConstants::TYPE_ATTRACTION,
                'attributes' => [ $this->attributeFactory->make([ 'name' => 'tag', 'data' => 'foo' ]) ],
            ]
        );
        $this->articleFactory->create(
            [
                'name' => 'bar',
                'mainType' => ArticleConstants::TYPE_ATTRACTION,
                'attributes' => [ $this->attributeFactory->make([ 'name' => 'tag', 'data' => 'bar' ]) ],
            ]
        );
        $this->articleFactory->create(
            [
                'name' => 'foo&bar',
                'mainType' => ArticleConstants::TYPE_ATTRACTION,
                'attributes' => [
                    $this->attributeFactory->make([ 'name' => 'tag', 'data' => 'bar' ]),
                    $this->attributeFactory->make([ 'name' => 'tag', 'data' => 'foo' ]),
                ],
            ]
        );

        $tour = $this->articleFactory->create([ 'mainType' => ArticleConstants::TYPE_TOUR ]);

        $this->node->setProperty('articleType', ArticleConstants::TYPE_ATTRACTION);
        $this->node->setProperty('preselectedTags', [ 'tag:foo' ]);
        $this->persistNode($this->node);

        $result = $this->subject->serialize($this->node);
        $this->assertCount(2, $result);
        $names = array_map(
            function ($item) {
                return $item['title'];
            },
            $result
        );

        $this->assertContains('foo', $names);
        $this->assertNotContains('bar', $names);
        $this->assertContains('foo&bar', $names);
    }

    public function testAllowsFilteringArticlesByAttributes(): void
    {
        $this->articleFactory->create(
            [
                'name' => 'loop',
                'mainType' => ArticleConstants::TYPE_TOUR,
                'attributes' => [ $this->attributeFactory->make([ 'name' => 'properties', 'data' => 'loopTour' ]) ],
            ]
        );
        $this->articleFactory->create(
            [
                'name' => 'one way',
                'mainType' => ArticleConstants::TYPE_TOUR,
                'attributes' => [ $this->attributeFactory->make([ 'name' => 'properties', 'data' => 'oneWayTour' ]) ],
            ]
        );

        $this->node->setProperty('articleType', ArticleConstants::TYPE_TOUR);
        $this->node->setProperty('preselectedProperties', [ 'properties:loopTour' ]);
        $this->persistNode($this->node);

        $result = $this->subject->serialize($this->node);
        $this->assertCount(1, $result);
        $this->assertEquals('loop', array_values($result)[0]['title']);
    }

    public function testAllowsFilteringArticlesByCategories(): void
    {
        [ $first, $second ] = $this->categoryFactory->createMultiple(2);
        $this->articleFactory->create(
            [ 'mainType' => ArticleConstants::TYPE_TOUR, 'name' => 'related to first', 'categories' => [ $first ] ]
        );
        $this->articleFactory->create(
            [ 'mainType' => ArticleConstants::TYPE_TOUR, 'name' => 'related to second', 'categories' => [ $second ] ]
        );

        $this->node->setProperty('articleType', ArticleConstants::TYPE_TOUR);
        $this->node->setProperty('preselectedCategories', [ $first->getPersistenceObjectIdentifier() ]);
        $this->persistNode($this->node);

        $result = $this->subject->serialize($this->node);
        $this->assertCount(1, $result);
        $this->assertEquals('related to first', array_values($result)[0]['title']);
        $this->assertNotEquals('related to second', array_values($result)[0]['title']);
    }

    public function testIncludesAllDetailsNeededForMap(): void
    {
        $attraction = $this->articleFactory->create([ 'mainType' => ArticleConstants::TYPE_ATTRACTION ]);

        $this->node->setProperty('articleType', ArticleConstants::TYPE_ATTRACTION);
        $this->persistNode($this->node);

        $result = $this->subject->serialize($this->node);
        $this->assertIsValidSerializedMarker(array_values($result)[0]);
    }

    public function testNumberOfArticlesIsLimited(): void
    {
        $this->articleFactory->createMultiple(250, [ 'mainType' => ArticleConstants::TYPE_ATTRACTION ]);

        $this->node->setProperty('articleType', ArticleConstants::TYPE_ATTRACTION);
        $this->persistNode($this->node);

        $this->inject($this->subject, 'limit', 50);
        $result = $this->subject->serialize($this->node);
        $this->assertLessThan(250, \count($result));
        $this->assertCount(50, $result);
    }

}
