<?php declare(strict_types=1);
namespace Newland\Toubiz\Map\Neos\ViewHelpers;

use Neos\ContentRepository\Domain\Model\Node;
use Neos\ContentRepository\Domain\Model\NodeInterface;
use Neos\Flow\Configuration\ConfigurationManager;
use Neos\Flow\Mvc\ActionRequest;
use Neos\FluidAdaptor\Core\ViewHelper\AbstractTagBasedViewHelper;
use Neos\Flow\Annotations as Flow;
use Newland\NeosIcons\IconUrlService;
use Newland\Toubiz\Map\Neos\Service\ApiKeyService;
use function Safe\json_encode;

class StaticMapViewHelper extends AbstractTagBasedViewHelper
{
    /**
     * @var ConfigurationManager
     * @Flow\Inject()
     */
    protected $configurationManager;
    /**
     * @var ApiKeyService
     * @Flow\Inject()
     */
    protected $apiKeyService;
    /**
     * @var IconUrlService
     * @Flow\Inject()
     */
    protected $iconUrlService;
    public function render(
        float $latitude,
        float $longitude,
        string $configurationPath,
        NodeInterface $node,
        int $width = null,
        int $height = null,
        int $zoom = null,
        string $mapType = null,
        ?array $geometry = null
    ): string {
        $staticMapConfiguration = (array) $this->configurationManager->getConfiguration(
            ConfigurationManager::CONFIGURATION_TYPE_SETTINGS,
            $configurationPath
        );
        $mapType = $mapType ?? $staticMapConfiguration['mapType'] ?? null;
        $width = $width ?? $staticMapConfiguration['width'] ?? 800;
        $height = $height ?? $staticMapConfiguration['height'] ?? 600;
        $zoom = $zoom ?? $staticMapConfiguration['zoom'] ?? 75;
        if ($mapType === null) {
            return '<!-- Static map not rendered: No map type defined. -->';
        }
        /** @var ActionRequest $request */
        $request = $this->controllerContext->getRequest();
        $iconUrl = $this->iconUrlService->buildMarkerIconUrl(
            $staticMapConfiguration['style'] ?? [],
            $request
        );
        $this->addApiKeysToTags($node);
        $this->addSourceSetAttributes($staticMapConfiguration);
        $this->tag->addAttributes(
            [
                'class' => 'cursor-pointer',
                'data-toubiz-static-map' => '',
                'data-toubiz-static-map.map-type' => $mapType,
                'data-toubiz-static-map.map.zoom' => $zoom,
                'data-toubiz-static-map.width' => $width,
                'data-toubiz-static-map.height' => $height,
                'data-toubiz-static-map.markers.0.icon-url' => $iconUrl,
                'data-toubiz-static-map.markers.0.latitude' => $latitude,
                'data-toubiz-static-map.markers.0.longitude' => $longitude,
            ]
        );
        if ($geometry) {
            $this->tag->addAttribute('data-toubiz-static-map.markers.0.tour', json_encode($geometry));
        }
        $this->tag->forceClosingTag(true);
        return $this->tag->render();
    }
    private function addApiKeysToTags(NodeInterface $node): void
    {
        $apiKeys = $this->apiKeyService->getApiKeys($node);
        if (array_key_exists('googleMaps', $apiKeys)) {
            $this->tag->addAttribute(
                'data-toubiz-static-map.map.google-maps-api-key',
                $apiKeys['googleMaps']
            );
        }
        if (array_key_exists('toursprung', $apiKeys)) {
            $this->tag->addAttribute(
                'data-toubiz-static-map.map.toursprung-api-key',
                $apiKeys['toursprung']
            );
        }
    }

    private function addSourceSetAttributes(array $staticMapConfiguration): void
    {
        foreach ($staticMapConfiguration['srcSet'] ?? [] as $index => $set) {
            $this->tag->addAttribute(
                sprintf('data-toubiz-static-map.map.src-set.%d.width', $index),
                '(int)' . $set['width']
            );
            $this->tag->addAttribute(
                sprintf('data-toubiz-static-map.map.src-set.%d.height', $index),
                '(int)' . $set['height']
            );

            if (array_key_exists('query', $set) && $set['query']) {
                $this->tag->addAttribute(
                    sprintf('data-toubiz-static-map.map.src-set.%d.query', $index),
                    '(string)' . $set['query']
                );
            }
        }
    }
}
