<?php declare(strict_types=1);

namespace Newland\Toubiz\Map\Neos\Serialization\Markers;

use Neos\ContentRepository\Domain\Model\NodeInterface;
use Neos\ContentRepository\Domain\Projection\Content\TraversableNodeInterface;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Mvc\ActionRequest;
use Neos\Media\Domain\Model\Image;
use Neos\Media\Domain\Model\ThumbnailConfiguration;
use Neos\Media\Domain\Service\AssetService;
use Neos\Neos\Domain\Service\ContentContextFactory;
use Newland\NeosCommon\Service\ControllerContextFactory;
use Newland\NeosIcons\IconUrlService;
use Newland\Toubiz\Map\Neos\Service\FusionSnippetRenderer;

class Links implements NodeSerializer
{
    protected const NODE_TYPE = 'Newland.Toubiz.Map.Neos:Map.Markers.Link';

    /**
     * @var FusionSnippetRenderer
     * @Flow\Inject()
     */
    protected $snippetRenderer;

    /**
     * @var AssetService
     * @Flow\Inject()
     */
    protected $assetService;

    /**
     * @var ControllerContextFactory
     * @Flow\Inject()
     */
    protected $contextFactory;

    /**
     * @var IconUrlService
     * @Flow\Inject()
     */
    protected $iconUrlService;

    /**
     * @var array
     * @Flow\InjectConfiguration(package="Newland.Toubiz.Map.Neos", path="markers.defaultIcons")
     */
    protected $defaultIconSettings;

    public function canSerialize(TraversableNodeInterface $node): bool
    {
        return $node->getNodeType()->isOfType(static::NODE_TYPE);
    }

    public function serialize(TraversableNodeInterface $node): array
    {
        if (!($node instanceof NodeInterface)) {
            return [];
        }

        $url = $this->resolveUri($node);
        if (!$url) {
            return [];
        }

        return [
            $node->getIdentifier() => [
                'paths' => [ (string) $node->findNodePath() ],
                'title' => $node->getProperty('title'),
                'categoryTitle' => $node->getProperty('categoryTitle'),
                'url' => $url,
                'image' => $this->resolveImage($node),
                'coordinates' => [
                    'longitude' => $node->getProperty('longitude'),
                    'latitude' => $node->getProperty('latitude'),
                ],
                'defaultStyle' => $this->getStyle($node),
                'tour' => null,
                'outdoorActiveTrackingId' => null,
                'detailApiEndpoints' => new \stdClass(),
            ],
        ];
    }

    private function resolveImage(NodeInterface $node): ?array
    {
        /** @var Image|null $imageModel */
        $imageModel = $node->getProperty('image');
        if ($imageModel === null) {
            return null;
        }

        $imageUri = $this->assetService->getThumbnailUriAndSizeForAsset(
            $imageModel,
            new ThumbnailConfiguration(),
            $this->contextFactory->initializeFakeActionRequest($node, [])
        )['src'] ?? null;

        return [
            'url' => $imageUri,
            'alt' => $imageModel->getTitle() ?: $node->getProperty('title'),
        ];
    }

    private function getStyle(NodeInterface $node): array
    {
        $style = [
            'icon' => $node->getProperty('icon') ?? $this->defaultIconSettings['icon'] ?? 'circle',
            'color' => $node->getProperty('color') ?? $this->defaultIconSettings['color'] ?? 'white',
            'backgroundColor' => $node->getProperty('backgroundColor')
                ?? $this->defaultIconSettings['color']
                ?? '#333',
        ];
        $style['iconUrl'] = $this->iconUrlService->buildMarkerIconUrlFromNode($node, $style);
        return $style;
    }

    private function resolveUri(NodeInterface $node): ?string
    {
        $uri = $node->getProperty('uri');
        if (!$uri) {
            return null;
        }

        $url = $this->snippetRenderer->renderSnippet(
            'lib.newland.toubiz.map.neos.convertUris',
            [ 'value' => $uri ],
            $node
        );
        if (!$url) {
            return null;
        }

        return $url;
    }
}
