<?php
namespace Newland\Toubiz\Map\Neos\Command;

/*
 * This file is part of the "toubiz-map-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\ContentRepository\Domain\Factory\NodeFactory;
use Neos\ContentRepository\Domain\Model\Node;
use Neos\ContentRepository\Domain\Model\NodeData;
use Neos\ContentRepository\Domain\Model\Workspace;
use Neos\ContentRepository\Domain\Repository\NodeDataRepository;
use Neos\ContentRepository\Domain\Repository\WorkspaceRepository;
use Neos\Eel\FlowQuery\FlowQuery;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Cli\CommandController;
use Neos\Neos\Domain\Model\Site;
use Neos\Neos\Domain\Repository\SiteRepository;
use Neos\Neos\Domain\Service\ContentContextFactory;
use Newland\NeosCommon\Domain\Repository\NodeRepository;
use Newland\Toubiz\Api\Constants\Language;
use Newland\Toubiz\Map\Neos\Controller\MapController;
use Newland\Toubiz\Map\Neos\Serialization\MapSerializer;
use Newland\Toubiz\Map\Neos\Serialization\SerializationService;
use Newland\Toubiz\Map\Neos\Service\ArticleCacheService;
use Newland\Toubiz\Map\Neos\Service\CategoryService;
use Newland\NeosCommon\Service\ControllerContextFactory;
use Newland\Toubiz\Sync\Neos\Domain\Repository\ArticleRepository;
use Symfony\Component\Console\Helper\ProgressBar;

/**
 * Map command controller.
 *
 * Provides commands to manipulate map data.
 *
 * @Flow\Scope("singleton")
 */
class MapCommandController extends CommandController
{
    const NODE_TYPE = 'Newland.Toubiz.Map.Neos:Map';

    /**
     * @var SerializationService
     * @Flow\Inject()
     */
    protected $serializationService;

    /**
     * @Flow\Inject()
     * @var NodeDataRepository
     */
    protected $nodeDataRepository;

    /**
     * @var WorkspaceRepository
     * @Flow\Inject()
     */
    protected $workspaceRepository;

    /**
     * @var SiteRepository
     * @Flow\Inject()
     */
    protected $siteRepository;

    /**
     * @var NodeFactory
     * @Flow\Inject()
     */
    protected $nodeFactory;

    /**
     * @var ContentContextFactory
     * @Flow\Inject()
     */
    protected $contentContextFactory;


    /**
     * Command to build an article cache to feed the map.
     *
     * As the map is based on a lot of marker data which requires a lot of
     * computing resources when fetching on-demand, static data is
     * built to assign to the view. This reduces server load.
     *
     * @return void
     */
    public function buildArticleCacheCommand()
    {
        foreach ($this->siteNodes() as $node) {
            /** @var Node $node */
            $bar = new ProgressBar($this->output->getOutput());
            $bar->setFormat('%current% [%bar%] ' . $node->getPath() . '@' . $node->getWorkspace()->getName());
            $this->serializationService->forceRegenerateAllCacheEntriesOfSubNodes(
                $node,
                function ($current, $total) use ($bar) {
                    $bar->setMaxSteps($total);
                    $bar->setProgress($current);
                }
            );
            $bar->finish();
            $this->outputLine();
        }
    }

    private function siteNodes()
    {
        /** @var Workspace[] $workspaces */
        $workspaces = $this->workspaceRepository->findAll();
        foreach ($workspaces as $workspace) {
            /** @var NodeData[] $nodeData */
            $nodeData = $this->nodeDataRepository->findByParentWithoutReduce('/sites', $workspace);
            foreach ($nodeData as $data) {
                yield $this->nodeFactory->createFromNodeData(
                    $data,
                    $this->contentContextFactory->create(
                        [
                            'workspaceName' => $data->getWorkspace()->getName(),
                            'invisibleContentShown' => false,
                            'inaccessibleContentShown' => false,
                            'removedContentShown' => false,
                            'dimensions' => $data->getDimensionValues(),
                        ]
                    )
                );
            }
        }
    }
}
