<?php declare(strict_types=1);

namespace Newland\Toubiz\Map\Neos\Tests\Unit\Serialization;

use Doctrine\ORM\EntityManagerInterface;
use Neos\ContentRepository\Domain\Model\NodeInterface;
use Neos\ContentRepository\Domain\Model\Workspace;
use Neos\ContentRepository\Domain\Repository\NodeDataRepository;
use Neos\ContentRepository\Domain\Utility\NodePaths;
use Neos\Neos\Service\PublishingService;
use Neos\Flow\Tests\FunctionalTestCase;
use Neos\Neos\Domain\Service\ContentContext;
use Newland\NeosTestingHelpers\InteractsWithNodes;
use Newland\Toubiz\Map\Neos\Serialization\MapSerializer;
use Newland\Toubiz\Map\Neos\Serialization\NodeSlot;
use Newland\Toubiz\Map\Neos\Serialization\SerializationService;
use PHPUnit\Framework\MockObject\MockObject;
use Neos\Neos\Ui\ContentRepository\Service\NodeService;
use Ramsey\Uuid\Uuid;

class NodeSlotTest extends FunctionalTestCase
{
    protected static $testablePersistenceEnabled = true;
    use InteractsWithNodes;

    /** @var NodeSlot */
    protected $subject;

    /** @var NodeInterface */
    protected $node;

    /** @var string */
    protected $randomName;

    /** @var MockObject<MapSerializer> */
    protected $serializer;

    public function setUp()
    {
        parent::setUp();
        $this->subject = $this->objectManager->get(NodeSlot::class);
        $this->randomName = md5(random_bytes(32));

        $this->serializer = $this->createMock(MapSerializer::class);
        $serializationService = $this->objectManager->get(SerializationService::class);
        $this->inject($serializationService, 'mapSerializer', $this->serializer);
        $this->inject($this->subject, 'serializationService', $serializationService);
    }

    public function tearDown(): void
    {
        $this->objectManager->forgetInstance(MapSerializer::class);
        $this->objectManager->forgetInstance(SerializationService::class);
        $this->objectManager->forgetInstance(NodeSlot::class);
        parent::tearDown();
    }

    public function testCallsSerializerForNodeInTargetWorkspace(): void
    {
        [ $node, $nodeInTargetWorkspace ] = $this->createMapNodes([ 'user-foo', 'live' ]);

        $this->serializer
            ->expects($this->once())
            ->method('serializeFilterItems')
            ->with($nodeInTargetWorkspace);
        $this->serializer
            ->expects($this->once())
            ->method('serializeMarkers')
            ->with($nodeInTargetWorkspace);

        $this->publish($node);
    }

    public function testDoesNothingIfNodeDoesNotExistInTargetWorkspace(): void
    {
        [ $node ] = $this->createMapNodes([ 'user-foo' ]);

        $this->serializer->expects($this->never())->method('serializeFilterItems');
        $this->serializer->expects($this->never())->method('serializeMarkers');

        $this->subject->onNodePublished($node, $this->getWorkspace('live'));
    }

    /** @dataProvider provideMapSubNodeTypes */
    public function testCallsSerializerForParentMapNodeIfNotMapDirectly(string $nodeType): void
    {
        [ $mapNode, $mapNodeInTargetWorkspace ] = $this->createMapNodes([ 'user-foo', 'live' ]);
        [ $child, $childNodeInTargetWorkspace ] = $this->createMapChildNodes([ 'user-foo', 'live' ], $nodeType);

        $this->serializer
            ->expects($this->once())
            ->method('serializeFilterItems')
            ->with($mapNodeInTargetWorkspace);
        $this->serializer
            ->expects($this->once())
            ->method('serializeMarkers')
            ->with($mapNodeInTargetWorkspace);

        $this->publish($child);
    }

    /** @dataProvider provideMapSubNodeTypes */
    public function testDoesNothingIfNoMapParentExists(string $nodeType): void
    {
        $child = $this->initializeNode('/test-site/testing', 'user-foo', null, [ ], null, $nodeType);

        $this->serializer->expects($this->never())->method('serializeFilterItems');
        $this->serializer->expects($this->never())->method('serializeMarkers');

        $this->publish($child);
    }

    public function testAlsoSerializesNewChildrenNode(): void
    {
        // For this test we are not mocking the serializer
        $serializer = $this->objectManager->get(MapSerializer::class);
        $service = $this->objectManager->get(SerializationService::class);
        $this->inject($service, 'mapSerializer', $serializer);
        $this->inject($this->subject, 'serializationService', $service);

        [ $mapNode, $mapNodeInTargetWorkspace ] = $this->createMapNodes([ 'user-foo', 'live' ]);
        [ $child ] = $this->createMapChildNodes([ 'user-foo' ], 'Newland.Toubiz.Map.Neos:Map.FilterItem');

        $published = $this->publish($mapNode);

        $items = $service->serializeAndCacheFilterItems($mapNodeInTargetWorkspace);
        $this->assertCount(1, $items);
    }

    public function testDoesNotTaintRepository(): void
    {
        [ $mapNode, $mapNodeInTargetWorkspace ] = $this->createMapNodes([ 'user-foo', 'live' ]);
        [ $child ] = $this->createMapChildNodes([ 'user-foo' ], 'Newland.Toubiz.Map.Neos:Map.FilterItem');

        $this->objectManager->get(NodeDataRepository::class)->remove($child);
        $this->publish($mapNode);
        $this->objectManager->get(NodeDataRepository::class)->remove($child);

        // If this test fails then doctrine throws an exception on the second `remove` call.
        $this->assertTrue(true);
    }

    public function provideMapSubNodeTypes(): array
    {
        return [
            [ 'Newland.Toubiz.Map.Neos:Map.FilterItem' ],
            [ 'Newland.Toubiz.Map.Neos:Map.Markers.FilteredArticles' ],
            [ 'Newland.Toubiz.Map.Neos:Map.Markers.Link' ],
            [ 'Newland.Toubiz.Map.Neos:Map.Markers.SelectedArticles' ],
            [ 'Newland.Toubiz.Map.Neos:Map.Markers.SelectedPages' ],
        ];
    }

    private function publish(NodeInterface $node, string $targetWorkspaceName = 'live'): NodeInterface
    {
        $targetWorkspace = $this->getWorkspace($targetWorkspaceName);
        $workspace = $node->getWorkspace();
        $workspace->setBaseWorkspace($targetWorkspace);

        $nodeService = $this->objectManager->get(NodeService::class);
        $nodeInCorrectContext = $nodeService->getNodeFromContextPath($node->getContextPath(), null, null, true);

        $this->objectManager->get(PublishingService::class)->publishNode($nodeInCorrectContext, $targetWorkspace);
        $this->subject->afterPublishing();

        return $nodeInCorrectContext;
    }

    /**
     * @param string[] $workspaces
     * @return NodeInterface[]
     */
    private function createMapNodes(array $workspaces): array
    {
        $identifier = Uuid::uuid4()->toString();
        return array_map(
            function(string $workspace) use ($identifier) {
                return $this->initializeNode(
                    sprintf('/test-site/%s', $this->randomName),
                    $workspace,
                    null,
                    [ ],
                    null,
                    'Newland.Toubiz.Map.Neos:Map',
                    null,
                    $identifier
                );
            },
            $workspaces
        );
    }

    /** @return NodeInterface[] */
    private function createMapChildNodes(array $workspaces, string $nodeType, string $subPath = 'foo')
    {
        $collectionPath = sprintf('/test-site/%s/some-collection', $this->randomName);
        $collectionIdentifier = Uuid::uuid4()->toString();
        $path = sprintf('/test-site/%s/some-collection/%s', $this->randomName, $subPath);
        $identifier = Uuid::uuid4()->toString();

        return array_map(
            function(string $workspace) use ($path, $collectionPath, $identifier, $collectionIdentifier, $nodeType) {
                $this->initializeNode($collectionPath, $workspace, null, [ ], null, 'Neos.Neos:ContentCollection', null, $collectionIdentifier);
                return $this->initializeNode($path, $workspace, null, [ ], null, $nodeType, null, $identifier);
            },
            $workspaces
        );
    }
}
