<?php declare(strict_types=1);

namespace Newland\Toubiz\Map\Neos\Serialization\Markers;

use Neos\ContentRepository\Domain\Projection\Content\TraversableNodeInterface;
use Newland\NeosCommon\Service\NodeService;
use Newland\Toubiz\Sync\Neos\Domain\Repository\ArticleRepository;
use Neos\Flow\Annotations as Flow;

/**
 * @Flow\Scope("singleton")
 */
class SelectedArticles implements NodeSerializer
{
    use ArticleFormatting;
    protected const NODE_TYPE = 'Newland.Toubiz.Map.Neos:Map.Markers.SelectedArticles';

    /**
     * @var ArticleRepository
     * @Flow\Inject()
     */
    protected $articleRepository;

    /**
     * @var NodeService
     * @Flow\Inject()
     */
    protected $nodeService;

    public function canSerialize(TraversableNodeInterface $node): bool
    {
        return $node->getNodeType()->isOfType(static::NODE_TYPE);
    }

    public function serialize(TraversableNodeInterface $node): array
    {
        $articleIds = $node->getProperty('articles') ?: [];

        $markers = [];
        if (!empty($articleIds)) {
            $markers = $this->buildMarkersFromArticleIds($articleIds, $node);
        }

        return $markers;
    }

    protected function buildMarkersFromArticleIds(array $articleIds, TraversableNodeInterface $node): array
    {
        $alwaysDisplayPolygon = (bool) $node->getProperty('alwaysDisplayPolygon');
        $color = $node->getProperty('tourColor');

        $language = $this->nodeService->getLanguage($node);
        $articles = $this->articleRepository->withLanguage(
            $language,
            function () use ($articleIds) {
                return $this->articleRepository->findByIdentifiers($articleIds);
            }
        );

        $markers = [];
        foreach ($articles as $article) {
            $markers[$article->getPersistenceObjectIdentifier()] = $this->articleToMarker(
                $article,
                $node,
                [
                    'tour' => [
                        'alwaysDisplayPolygon' => $alwaysDisplayPolygon,
                        'color' => $color,
                    ],
                ]
            );
        }
        return $markers;
    }
}
