<?php
namespace Newland\Toubiz\Map\Neos\Command;

/*
 * This file is part of the "toubiz-map-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\ContentRepository\Domain\Model\Node;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Cli\CommandController;
use Newland\NeosCommon\Domain\Repository\NodeRepository;
use Newland\Toubiz\Map\Neos\Controller\MapController;
use Newland\Toubiz\Map\Neos\Service\ArticleCacheService;
use Newland\Toubiz\Map\Neos\Service\CategoryService;
use Newland\NeosCommon\Service\ControllerContextFactory;
use Newland\Toubiz\Sync\Neos\Domain\Repository\ArticleRepository;

/**
 * Map command controller.
 *
 * Provides commands to manipulate map data.
 *
 * @Flow\Scope("singleton")
 */
class MapCommandController extends CommandController
{
    const NODE_TYPE = 'Newland.Toubiz.Map.Neos:Map';

    /**
     * @var ArticleRepository
     * @Flow\Inject
     */
    protected $articleRepository;

    /**
     * @Flow\InjectConfiguration(package="Newland.Toubiz.Map.Neos")
     * @var array
     */
    protected $configuration;

    /**
     * @Flow\Inject
     * @var ArticleCacheService
     */
    protected $articleCache;

    /**
     * @Flow\Inject
     * @var CategoryService
     */
    protected $categoryService;

    /**
     * @Flow\Inject
     * @var NodeRepository
     */
    protected $nodeRepository;

    /**
     * @Flow\Inject
     * @var ControllerContextFactory
     */
    protected $contextFactory;

    /**
     * Command to build an article cache to feed the map.
     *
     * As the map is based on a lot of marker data which requires a lot of
     * computing resources when fetching on-demand, static data is
     * built to assign to the view. This reduces server load.
     *
     * @return void
     */
    public function buildArticleCacheCommand()
    {
        $nodes = $this->nodeRepository
            ->findNodesByNodeType(static::NODE_TYPE, true);
        $this->output->progressStart(\count($nodes));
        $this->categoryService->setConfiguration($this->configuration);

        foreach ($nodes as $index => $node) {
            $this->output->progressSet($index + 1);
            $this->output->output('   Warming cache for ' . $this->nodeLabel($node) . "\r");

            $articleJson = $this->buildMapCache($node);
            $this->buildCategoryCache($node, $articleJson);
        }

        $this->output->progressFinish();
        $this->outputLine("\n");
    }

    private function buildMapCache(Node $node): string
    {
        $this->articleRepository->setLanguage($node->getDimensions()['language'][0] ?? null);
        $query = $this->articleRepository->queryForFilter(
            MapController::createArticleFilter((array) $node->getProperties(), $this->configuration)
        );

        return $this->articleCache->generateAndStoreInCache(
            $query,
            $this->contextFactory->initializeFakeControllerContext($node)
        );
    }

    private function buildCategoryCache(Node $node, string $articleJson)
    {
        $this->categoryService->setProperties((array) $node->getProperties());
        $this->categoryService->generateAndStoreInCache($articleJson);
    }

    private function nodeLabel(Node $node): string
    {
        return $node->getName() . json_encode($node->getDimensions());
    }
}
