<?php
namespace Newland\Toubiz\Map\Neos\Controller;

/*
 * This file is part of the "toubiz-map-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\ContentRepository\Domain\Model\Node;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Mvc\Controller\ActionController;
use Neos\Flow\Mvc\View\ViewInterface;
use Neos\Neos\Domain\Service\ContentContext;
use Newland\NeosCommon\Service\ConfigurationService;
use Newland\Toubiz\Map\Neos\DataSource\PageForMapDataSource;
use Newland\Toubiz\Map\Neos\Utility\CategoryMappingUtility;
use Newland\Toubiz\Sync\Neos\Domain\Filter\ArticleFilter;
use Newland\Toubiz\Sync\Neos\Domain\Repository\ArticleRepository;
use Newland\Toubiz\Map\Neos\Service\ArticleCacheService;
use Newland\Toubiz\Map\Neos\Service\CategoryService;

/**
 * Map controller.
 *
 * @Flow\Scope("singleton")
 */
class MapController extends ActionController
{
    /**
     * @var ArticleRepository
     * @Flow\Inject
     */
    protected $articleRepository;

    /**
     * @Flow\InjectConfiguration
     * @var array
     */
    protected $configuration;

    /**
     * @var array
     */
    protected $properties = [];

    /**
     * @Flow\Inject
     * @var ArticleCacheService
     */
    protected $articleCacheService;

    /**
     * @Flow\Inject
     * @var CategoryService
     */
    protected $categoryService;

    /**
     * @Flow\InjectConfiguration(package="Newland.Toubiz.Sync.Neos.services.Outdooractive/Api")
     * @var array
     */
    protected $outdooractiveCredentials;

    /**
     * @Flow\Inject
     * @var PageForMapDataSource
     */
    protected $pageDataSource;

    /**
     * @var ConfigurationService
     * @Flow\Inject
     */
    protected $configurationService;

    /**
     * @var Node
     */
    protected $node;

    public static function createArticleFilter(array $properties, array $configuration): ArticleFilter
    {
        $articleFilter = new ArticleFilter;
        if (!empty($properties['categories'])) {
            $categoryIds = CategoryMappingUtility::getMappedIdsFromConfiguration(
                $configuration['categories'],
                $properties['categories']
            );

            $articleFilter->setCategoriesIdentifierField('originalId');
            $articleFilter->setCategories($categoryIds);
        }
        if (!empty($properties['cities'])) {
            $articleFilter->setZips($properties['cities']);
        }
        if (!empty($properties['articles'])) {
            $articleFilter->setIdentifiers($properties['articles']);
        }
        $articleFilter->setExcludeUnsafeCoordinates(true);
        return $articleFilter;
    }


    /**
     * Action initializer.
     *
     * @return void
     */
    public function initializeAction()
    {
        /** @var Node|null $node */
        $node = $this->request->getInternalArgument('__node');
        if ($node instanceof Node) {
            $this->node = $node;
        }

        if ($this->node !== null) {
            $this->properties = (array) $this->node->getProperties();
        }

        $this->categoryService->setProperties($this->properties);
        $this->categoryService->setConfiguration($this->configuration);
    }

    protected function initializeView(ViewInterface $view)
    {
        $view->assignMultiple($this->properties);
    }

    /**
     * Show action.
     *
     * Displays the map.
     *
     * @return void
     */
    public function showAction()
    {
        $context = $this->node->getContext();
        $isInBackend = ($context instanceof ContentContext) && $context->isInBackend();

        $articleJson = '[]';
        $nodes = [];
        if (!$isInBackend) {
            if ($this->shouldBeDisplayed('articles')) {
                $filter = static::createArticleFilter($this->properties, $this->configuration);
                $articleJson = $this->articleCacheService->cacheQueryResultsAsJson(
                    $this->articleRepository->queryForFilter($filter),
                    $this->controllerContext
                );
            }

            if ($this->shouldBeDisplayed('pageNodes')) {
                $nodes = $this->pageDataSource->pagesForMap($this->node, $this->properties['pages'] ?? []);
            }
        }

        $categories = $this->categoryService->getFilteredCategories($articleJson);

        $this->view->assignMultiple(
            [
                'configuration' => $this->configuration,
                'apiKey' => $this->getApiKey(),
                'properties' => $this->properties,
                'node' => $this->node,
                'nodes' => $nodes,
                'categories' => $categories,
                'articleJson' => $articleJson,
                'outdooractiveCredentials' => $this->outdooractiveCredentials,
            ]
        );
    }

    /**
     * Helper method to evaluate if code has to be executed
     * based on the configuration type limitation.
     *
     * @param string $type
     * @return bool
     */
    protected function shouldBeDisplayed($type)
    {
        if (empty($this->properties['limitToTypes'])) {
            return true;
        }

        return in_array($type, $this->properties['limitToTypes']);
    }

    private function getApiKey()
    {
        $apiKey = $this->configurationService->getCurrentSiteConfiguration(
            $this->node,
            'Newland.Toubiz.Map.Neos.services.map.apiKey'
        );

        if (empty($apiKey)) {
            // legacy configuration
            $apiKey = $this->configurationService->getConfiguration('Newland.Toubiz.Map.Neos.apiKey');
        }

        return $apiKey;
    }
}
