<?php
namespace Newland\Toubiz\Map\Neos\Service;

use Newland\Toubiz\Map\Neos\Utility\CategoryMappingUtility;
use Neos\Flow\Cache\CacheManager;
use Neos\Flow\Annotations as Flow;

class CategoryService
{
    const CACHE_NAME = 'Newland_Toubiz_Map_Neos-Data';

    /**
     * @var array
     */
    protected $properties = [];

    /**
     * @var array
     */
    protected $configuration = [];

    /**
    * @Flow\Inject
    * @var CacheManager
    */
    protected $cacheManager;

    /**
     * @param array $properties
     */
    public function setProperties(array $properties)
    {
        $this->properties = $properties;
    }

    /**
     * @param array $configuration
     */
    public function setConfiguration(array $configuration)
    {
        $this->configuration = $configuration;
    }

    /**
     * Returns all used categories and subcategories.
     *
     * Present values are taken from the cache.
     * Otherwise a new entry will be created and returned.
     *
     * @param string $articleJson
     * @return array|mixed
     * @throws \Neos\Cache\Exception\NoSuchCacheException
     */
    public function getFilteredCategories(string $articleJson)
    {
        $cache = $this->cacheManager->getCache(static::CACHE_NAME);
        $cacheKey = $this->cacheKey($articleJson);

        if ($cache->has($cacheKey)) {
            return $cache->get($cacheKey);
        }
        return $this->generateAndStoreInCache($articleJson);
    }

    /**
     * @param string $articleJson
     * @return array
     * @throws \Neos\Cache\Exception\NoSuchCacheException
     */
    public function generateAndStoreInCache(string $articleJson)
    {
        $cache = $this->cacheManager->getCache(static::CACHE_NAME);
        $categories = $this->generateFilteredCategories(json_decode($articleJson, true));
        $cache->set($this->cacheKey($articleJson), $categories);
        return $categories;
    }

    /**
     * @param string $articleJson
     * @return string
     */
    private function cacheKey(string $articleJson): string
    {
        return md5($articleJson . json_encode($this->properties) . json_encode($this->configuration));
    }

    /**
     * Compare requested with given points.
     *
     * Unused categories and subcategories will be removed.
     *
     * @param array $articles
     * @return array
     */
    private function generateFilteredCategories(array $articles)
    {
        $points = [];
        $categories = $this->getCategories();

        foreach ($articles as $article) {
            $points[] = $article['category'];
        }

        foreach ($categories as $parentIndex => $parentCategory) {
            foreach ($parentCategory['children'] as $childIndex => $childCategory) {
                if (!array_key_exists('map', $childCategory)
                    || \count(array_intersect($points, $childCategory['map'])) === 0
                ) {
                    unset($categories[$parentIndex]['children'][$childIndex]);
                }
            }

            if ($categories[$parentIndex]['children'] === []) {
                unset($categories[$parentIndex]);
            }
        }

        return $categories;
    }

    /**
     * Returns configured categories.
     *
     * Optionally filters configured categories based on
     * the given sub-configuration (i.e. from a node).
     *
     * @return array
     */
    protected function getCategories()
    {
        if (!array_key_exists('categories', $this->configuration)) {
            // No search without a haystack.
            return [];
        }

        if (array_key_exists('categories', $this->properties)
            && !empty($this->properties['categories'])
        ) {
            return CategoryMappingUtility::getCategoriesFromConfiguration(
                $this->configuration['categories'],
                $this->properties['categories']
            );
        } else {
            // No filtering without needles.
            return $this->configuration['categories'];
        }
    }
}
