<?php
namespace Newland\Toubiz\Events\Neos\Controller;

/*
 * This file is part of the "toubiz-events-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\ContentRepository\Domain\Model\Node;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\I18n\Locale;
use Neos\Flow\I18n\Service;
use Neos\Flow\Mvc\Controller\ActionController;
use Neos\Flow\Mvc\View\ViewInterface;
use Neos\Neos\Service\LinkingService;
use Newland\NeosCommon\Service\NodeService;

/**
 * Abstract action controller.
 *
 * Abstract action controller for this package's controllers.
 *
 * @Flow\Scope("singleton")
 */
abstract class AbstractActionController extends ActionController
{
    /**
     * The current document node (page node).
     *
     * @var \Neos\ContentRepository\Domain\Model\Node
     */
    protected $documentNode;

    /**
     * The current node (mostly the plugin).
     *
     * @var \Neos\ContentRepository\Domain\Model\Node
     */
    protected $node;

    /**
     * The integration type.
     *
     * @var string
     */
    protected $integrationType;

    /**
     * @Flow\Inject
     * @var LinkingService
     */
    protected $linkingService;

    /**
     * @var array
     */
    protected $properties = [];

    /**
     * Package-specific settings.
     *
     * @Flow\InjectConfiguration(package="Newland.Toubiz.Events.Neos")
     * @var array
     */
    protected $settings;

    /**
     * @Flow\Inject
     * @var Service
     */
    protected $i18nService;

    /**
     * @var NodeService
     * @Flow\Inject()
     */
    protected $nodeService;

    /**
     * Action initializer.
     *
     * @return void
     */
    public function initializeAction()
    {
        $this->documentNode = $this->request->getInternalArgument('__documentNode');
        $this->node = $this->request->getInternalArgument('__node');
        $this->integrationType = $this->settings['integrationType'];

        if ($this->node && $this->node->getProperties()) {
            $this->properties = $this->node->getProperties();
        }
    }

    /**
     * View initializer.
     *
     * @param ViewInterface $view
     * @return void
     */
    public function initializeView(ViewInterface $view)
    {
        $view->assignMultiple(
            [
                'documentNode' => $this->documentNode,
                'integrationType' => ucwords($this->integrationType),
                'node' => $this->node,
                'action' => $this->request->getControllerActionName(),
            ]
        );
    }

    protected function setupCustomNodeMappingForAjaxRequest(Node $node)
    {
        $this->request->setArgument('__node', $node);
        $this->request->setArgument('__documentNode', $this->nodeService->getDocumentNode($node));

        $this->initializeAction();
        $this->initializeView($this->view);

        $this->applyCustomLanguageMappingFromDimensions($node);
    }

    private function applyCustomLanguageMappingFromDimensions(Node $node)
    {
        $dimensions = $node->getContext()->getDimensions();
        if (array_key_exists('language', $dimensions) && $dimensions['language'] !== []) {
            $configuration = $this->i18nService->getConfiguration();
            $currentLocale = new Locale($dimensions['language'][0]);
            $configuration->setCurrentLocale($currentLocale);
            $configuration->setFallbackRule(
                [
                    'strict' => false,
                    'order' => array_reverse($dimensions['language']),
                ]
            );
        }
    }
}
