<?php declare(strict_types=1);
namespace Newland\Toubiz\Events\Neos\Controller;

/*
 * This file is part of the "toubiz-events-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\ContentRepository\Domain\Model\Node;
use Neos\ContentRepository\Domain\Service\ContextFactoryInterface;
use Neos\ContentRepository\Exception\PageNotFoundException;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Mvc\Routing\ObjectPathMapping;
use Neos\Flow\Mvc\Routing\ObjectPathMappingRepository;
use Neos\Flow\Utility\Now;
use Neos\Neos\Domain\Service\ContentContext;
use Newland\NeosCommon\Logging\DeprecationLog;
use Newland\Toubiz\Events\Neos\Service\EventUrlService;
use Newland\Toubiz\Sync\Neos\Domain\Model\Event;
use Newland\Toubiz\Sync\Neos\Domain\Repository\EventRepository;

class RedirectController extends AbstractActionController
{

    /**
     * @var EventUrlService
     * @Flow\Inject()
     */
    protected $eventUrlService;

    /**
     * @var ObjectPathMappingRepository
     * @Flow\Inject()
     */
    protected $objectPathMappingRepository;

    /**
     * @var EventRepository
     * @Flow\Inject()
     */
    protected $eventRepository;

    /**
     * @var ContextFactoryInterface
     * @Flow\Inject()
     */
    protected $contextFactory;

    /**
     * @var DeprecationLog
     * @Flow\Inject()
     */
    protected $deprecationLog;


    public function eventDetailAction(string $uriSegment): void
    {
        $event = $this->findPossibleRedirectionTarget($uriSegment);

        $node = $this->getCurrentSiteNode($event->getLanguage());
        if ($node === null) {
            throw new PageNotFoundException(sprintf(
                'No site node can be resolved for event %s',
                $event->getPersistenceObjectIdentifier()
            ));
        }

        $uri = $this->eventUrlService->generateUrlByCurrentNode($event, $node);
        $this->deprecationLog->logDeprecation('Deprecated event URL ' . $uriSegment . ' redirected to ' . $uri);
        $this->redirectToUri($uri, 0, 301);
    }

    private function findPossibleRedirectionTarget(string $uriSegment): Event
    {
        $event = $this->findEventBasedOnIncorrectPath($uriSegment)
            ?? $this->findEventBasedOnMissingUrlIdentifier($uriSegment)
            ?? $this->findEventBasedOnPersistenceObjectIdentifier($uriSegment)
            ?? $this->findEventBasedOnOriginalId($uriSegment);

        if ($event === null) {
            throw new PageNotFoundException(sprintf('Uri segment %s could not be resolved to event', $uriSegment));
        }

        return $event;
    }

    private function findEventBasedOnIncorrectPath(string $uriSegment): ?Event
    {
        $parts = (array) preg_split('/\W/', $uriSegment);
        $potentialUrlIdentifier = $parts[count($parts) - 1];
        return $this->eventRepository->withoutLanguageHandling(function () use ($potentialUrlIdentifier) {
            return $this->eventRepository->findOneBy([ 'urlIdentifier' => $potentialUrlIdentifier ]);
        });
    }

    private function findEventBasedOnMissingUrlIdentifier(string $uriSegment): ?Event
    {
        $query = $this->objectPathMappingRepository->createQuery();

        $parts = [];
        $parts[] = $query->logicalAnd([
            $query->equals('objectType', Event::class),
            $query->equals('uriPattern', '{title}-{urlIdentifier}'),
            $query->like('pathSegment', $uriSegment . '%'),
        ]);
        $parts[] = $query->logicalAnd([
            $query->equals('objectType', Event::class),
            $query->equals('uriPattern', '{urlIdentifier}-{title}'),
            $query->like('pathSegment', '%' . $uriSegment),
        ]);

        $query->matching($query->logicalOr($parts));
        $query->setLimit(1);

        /** @var ObjectPathMapping|null $mapping */
        $mapping = $query->execute()->getFirst();
        if ($mapping === null) {
            return null;
        }

        return $this->eventRepository->withoutLanguageHandling(function () use ($mapping) {
            return $this->eventRepository->findByIdentifier($mapping->getIdentifier());
        });
    }

    private function findEventBasedOnPersistenceObjectIdentifier(string $uriSegment): ?Event
    {
        return $this->eventRepository->withoutLanguageHandling(function () use ($uriSegment) {
            return $this->eventRepository->findByIdentifier($uriSegment);
        });
    }

    private function findEventBasedOnOriginalId(string $uriSegment): ?Event
    {
        return $this->eventRepository->withoutLanguageHandling(function () use ($uriSegment) {
            return $this->eventRepository->findOneByOriginalId($uriSegment);
        });
    }


    protected function getCurrentSiteNode(?string $language): ?Node
    {
        $dimensions = $language ? [ 'language' => [ $language ] ] : [];
        $context = $this->contextFactory->create(
            [
                'workspaceName' => 'live',
                'currentDateTime' => new Now(),
                'dimensions' => $dimensions,
                'invisibleContentShown' => false,
                'removedContentShown' => false,
                'inaccessibleContentShown' => false,
            ]
        );

        if (!($context instanceof ContentContext)) {
            return null;
        }

        $node = $context->getCurrentSiteNode();
        if (!($node instanceof Node)) {
            return null;
        }

        return $node;
    }
}
