<?php
namespace Newland\Toubiz\Events\Neos\Controller;

/*
 * This file is part of the "toubiz-events-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Neos\Flow\Property\Exception\TargetNotFoundException;
use Neos\Flow\Property\TypeConverter\Error\TargetNotFoundError;
use Neos\FluidAdaptor\View\StandaloneView;
use Newland\Toubiz\Api\ObjectAdapter\EventAdapterInterface;
use Newland\Toubiz\Events\Neos\Filter\EventDateFilterFactory;
use Newland\Toubiz\Sync\Neos\Domain\Model\Event;
use Newland\Toubiz\Sync\Neos\Domain\Repository\EventRepository;
use Newland\ToubizWidgetRendering\Renderer\ToubizWidgetRenderer;

class EventsController extends AbstractActionController
{
    /**
     * @var EventRepository
     * @Flow\Inject()
     */
    protected $eventRepository;

    /**
     * @var ToubizWidgetRenderer
     * @Flow\Inject()
     */
    protected $widgetRenderer;

    /**
     * @var array
     * @Flow\InjectConfiguration()
     */
    protected $configuration;

    public function teaserAction(): void
    {

        $this->view->assignMultiple(
            [
                'event' => $this->eventRepository->findByIdentifier($this->properties['event'] ?? ''),
                'wrapStart' => $this->request->getInternalArgument('__wrapStart'),
                'wrapEnd' => $this->request->getInternalArgument('__wrapEnd'),
                'slots' => $this->request->getInternalArgument('__slots'),
            ]
        );
    }

    /**
     * Teaser List action.
     *
     * Lists a subset of upcoming events.
     *
     * @return void
     */
    public function teaserListAction(): void
    {
        $today = (new \DateTime())->format('Y-m-d');

        $this->properties['numberOfDays'] = $this->getDateIntervalForTeaserList($this->properties['numberOfDays'] ?? 1);
        $filter = (new EventDateFilterFactory($this->node))
            ->createEventFilter($this->properties['eventType'] ?? [], $this->properties)
            ->initializeForFilteredLists($this->request);

        $eventDates = $filter
            ->getEventQuery()
            ->setMaxResults($this->properties['recordLimit'])
            ->getQuery()
            ->execute();

        $this->view->assignMultiple(
            [
                'wrapStart' => $this->request->getInternalArgument('__wrapStart'),
                'wrapEnd' => $this->request->getInternalArgument('__wrapEnd'),
                'currentDate' => $today,
                'eventDates' => $eventDates,
            ]
        );
    }

    /**
     * Teaser Slider action.
     *
     * Lists a subset of upcoming events in a slider.
     *
     * @return void
     */
    public function teaserSliderAction(): void
    {
        $today = (new \DateTime())->format('Y-m-d');

        $this->properties['numberOfDays'] = $this->getDateIntervalForTeaserList($this->properties['numberOfDays'] ?? 1);
        $filter = (new EventDateFilterFactory($this->node))
            ->createEventFilter($this->properties['eventType'] ?? [], $this->properties)
            ->initializeForFilteredLists($this->request);

        $eventDates = $filter
            ->getEventQuery()
            ->setMaxResults($this->properties['recordLimit'])
            ->getQuery()
            ->execute();

        $this->view->assignMultiple(
            [
                'wrapStart' => $this->request->getInternalArgument('__wrapStart'),
                'wrapEnd' => $this->request->getInternalArgument('__wrapEnd'),
                'currentDate' => $today,
                'eventDates' => $eventDates,
                'id' => uniqid(),
                'teaserStyle' => $this->properties['teaserStyle'] ?? '',
            ]
        );
    }

    private function getDateIntervalForTeaserList(int $numberOfDays = 1): array
    {
        $modify = \Safe\sprintf('+%d days', max(0, $numberOfDays - 1));
        $lastDay = (clone $this->now)->modify($modify);

        return [
            'from' => $this->now->format('Y-m-d'),
            'to' => $lastDay->format('Y-m-d'),
        ];
    }

    /**
     * @param Event $event
     * @return void|string
     */
    public function showAction(Event $event = null)
    {
        if (!$event) {
            return;
        }

        if ($event->getDataSource() === EventAdapterInterface::DATA_SOURCE_TOUBIZ_APIV1) {
            $view = new StandaloneView();
            $view->setControllerContext($this->controllerContext);
            $view->setTemplate('ShowWidget');
            $view->assignMultiple(
                [
                    'event' => $event,
                    'component' => 'tb-w-event-detail',
                ]
            );
            return $view->render();
        }

        $this->view->assign('event', $event);
    }

    protected function handleTargetNotFoundError(): void
    {
        foreach (array_keys($this->request->getArguments()) as $argumentName) {
            /** @var TargetNotFoundError $targetNotFoundError */
            $targetNotFoundError = $this->arguments
                ->getValidationResults()
                ->forProperty($argumentName)
                ->getFirstError(TargetNotFoundError::class);

            if ($targetNotFoundError !== false) {
                $this->redirectToUri($this->configuration['notFoundPage'] ?? '/404', 0, 404);
            }
        }
    }
}
