<?php declare(strict_types=1);

namespace Newland\Toubiz\Events\Tests\Unit\Filter;

use Neos\Flow\Tests\FunctionalTestCase;
use Neos\Flow\Utility\Now;
use Newland\Toubiz\Events\Neos\Filter\Items\EventDateRoot;
use Newland\Toubiz\Sync\Neos\Domain\Model\EventDate;
use Newland\Toubiz\Sync\Neos\Tests\Factory\EventDateFactory;
use Newland\Toubiz\Sync\Neos\Tests\Factory\EventFactory;

class EventDateRootTest extends FunctionalTestCase
{
    protected static $testablePersistenceEnabled = true;

    /** @var EventDateRoot */
    protected $subject;

    public function setUp(): void
    {
        parent::setUp();
        $this->subject = $this->objectManager->get(EventDateRoot::class);
    }

    // See TBEVENTS-133
    public function testFindsEventsThatStartedEarlierToday(): void
    {
        // Event is at 14:00 but it is 14:30 right now.
        $this->inject($this->subject, 'now', new Now('2019-02-02T14:30'));
        $date = $this->createEvent('2019-02-02T14:00', '2019-02-02T15:00');

        /** @var EventDate[] $dates */
        $dates = $this->subject->getEventQuery()->getQuery()->execute();
        $ids = array_map(function(EventDate $date) { return $date->getPersistenceObjectIdentifier(); }, $dates);

        $this->assertContains($date->getPersistenceObjectIdentifier(), $ids, 'Event that begins at 14:00 should be found if currently it is 14:30');
    }

    public function testDoesNotFindEventsThatWereTodayButHaveAlreadyFinished(): void
    {
        // Event ends at 12:00 but it is 14:30 right now.
        $this->inject($this->subject, 'now', new Now('2019-02-02T14:30'));
        $date = $this->createEvent('2019-02-02T10:00', '2019-02-02T12:00');

        /** @var EventDate[] $dates */
        $dates = $this->subject->getEventQuery()->getQuery()->execute();
        $ids = array_map(function(EventDate $date) { return $date->getPersistenceObjectIdentifier(); }, $dates);

        $this->assertNotContains($date->getPersistenceObjectIdentifier(), $ids, 'Event that ends at 12:00 should be found if currently it is 14:30');
    }

    // See TBEVENTS-133
    public function testFindsFullDayEventsToday(): void
    {
        $this->inject($this->subject, 'now', new Now('2019-02-02T14:30'));
        $date1 = $this->createEvent('2019-02-02T00:00', '2019-02-02T00:00');
        $date2 = $this->createEvent('2019-02-02T00:00', '2019-02-02T23:59');
        $date3 = $this->createEvent('2019-02-02T00:00', '2019-02-03T00:00');

        /** @var EventDate[] $dates */
        $dates = $this->subject->getEventQuery()->getQuery()->execute();
        $ids = array_map(function(EventDate $date) { return $date->getPersistenceObjectIdentifier(); }, $dates);

        $this->assertContains($date1->getPersistenceObjectIdentifier(), $ids, 'Full day event (00:00-00:00 same day) should be found');
        $this->assertContains($date2->getPersistenceObjectIdentifier(), $ids, 'Full day event (00:00-23:59 same day) should be found');
        $this->assertContains($date3->getPersistenceObjectIdentifier(), $ids, 'Full day event (00:00-00:00 next day) should be found');
    }

    public function testDoesNotFindEventsThatAreYesterday(): void
    {
        // Today is 2019-02-02 but event was on 2019-02-01
        $this->inject($this->subject, 'now', new Now('2019-02-02T14:30'));
        $date = $this->createEvent('2019-02-01T14:00', '2019-02-01T15:00');

        /** @var EventDate[] $dates */
        $dates = $this->subject->getEventQuery()->getQuery()->execute();
        $ids = array_map(function(EventDate $date) { return $date->getPersistenceObjectIdentifier(); }, $dates);

        $this->assertNotContains($date->getPersistenceObjectIdentifier(), $ids);
    }

    private function createEvent(string $from, string $to): EventDate
    {
        return (new EventDateFactory($this->objectManager))->create([
            'beginsAt' => $from,
            'endsAt' => $to,
            'event' => (new EventFactory($this->objectManager))->create(),
        ]);
    }
}
