<?php declare(strict_types=1);


namespace Newland\Toubiz\Events\Neos\Tests\Unit\Service;

use Neos\Flow\Tests\FunctionalTestCase;
use Newland\Toubiz\Events\Neos\Service\FilterConfiguration;

class FilterConfigurationTest extends FunctionalTestCase
{

    /** @var FilterConfiguration */
    protected $subject;

    public function setUp()
    {
        parent::setUp();
        $this->subject = new FilterConfiguration();
    }

    public function testItShouldNotAddSectionIfNoFieldSetsInIt(): void
    {
        $this->inject($this->subject, 'filterSections', [ 'foo' => [ 'fieldSets' => [] ] ]);
        $filterSections = $this->subject->getFilterSections([], []);
        $this->assertArrayNotHasKey('foo', $filterSections);
        $this->assertEmpty($filterSections);
    }

    public function testItShouldAddCategoriesToFilterSections(): void
    {
        $this->inject($this->subject, 'filterSections', [
            'foo' => [
                'fieldSets' => [
                    [ 'type' => 'categories' ]
                ]
            ]
        ]);
        $filterSections = $this->subject->getFilterSections([], []);
        $this->assertArrayHasKey('foo', $filterSections);
    }

    public function testItShouldNotAddCategoriesToFilterSectionsIfInQueryOverride(): void
    {
        $this->inject($this->subject, 'filterSections', [
            'foo' => [
                'fieldSets' => [
                    [ 'type' => 'categories' ]
                ]
            ]
        ]);
        $filterSections = $this->subject->getFilterSections([ 'categories' => [ 'abc' ] ], []);
        $this->assertArrayNotHasKey('foo', $filterSections);
    }

    public function testItShouldAddRegionsToFilterSections(): void
    {
        $this->inject($this->subject, 'filterSections', [
            'foo' => [
                'fieldSets' => [
                    [ 'type' => 'regions' ]
                ]
            ]
        ]);
        $filterSections = $this->subject->getFilterSections([], []);
        $this->assertArrayHasKey('foo', $filterSections);
    }

    public function testItShouldNotAddRegionsToFilterSectionsIfPreselectedRegionsInQueryOverride(): void
    {
        $this->inject($this->subject, 'filterSections', [
            'foo' => [
                'fieldSets' => [
                    [ 'type' => 'regions' ]
                ]
            ]
        ]);
        $filterSections = $this->subject->getFilterSections([ 'preselectedRegions' => [ 'abc' ] ], []);
        $this->assertArrayNotHasKey('foo', $filterSections);
    }

    public function testItShouldAddCheckboxesToFilterSections(): void
    {
        $this->inject($this->subject, 'filterSections', [
            'foo' => [
                'fieldSets' => [
                    [
                        'type' => 'checkboxes',
                        'items' => [ 'foo', 'bar', 'baz' ]
                    ]
                ]
            ]
        ]);
        $filterSections = $this->subject->getFilterSections([], [ 'preselectedTopics' => [] ]);
        $this->assertArrayHasKey('foo', $filterSections);
        $this->assertEquals([ 'foo', 'bar', 'baz' ], $filterSections['foo']['fieldSets'][0]['items']);
    }

    public function testItShouldNotAddCheckboxesToFilterSectionsIfNoItemsSpecified(): void
    {
        $this->inject($this->subject, 'filterSections', [
            'foo' => [
                'fieldSets' => [
                    [
                        'type' => 'checkboxes',
                        'items' => [ ]
                    ]
                ]
            ]
        ]);
        $filterSections = $this->subject->getFilterSections([], [ 'preselectedTopics' => [] ]);
        $this->assertArrayNotHasKey('foo', $filterSections);
    }

    public function testItShouldRemoveItemsIfTheyAreAlsoInQueryOverride(): void
    {
        $this->inject($this->subject, 'filterSections', [
            'foo' => [
                'fieldSets' => [
                    [
                        'type' => 'checkboxes',
                        'items' => [ 'foo', 'bar', 'baz' ]
                    ]
                ]
            ]
        ]);
        $filterSections = $this->subject->getFilterSections([], [ 'preselectedTopics' => [ 'baz' ] ]);
        $this->assertEquals([ 'foo', 'bar' ], $filterSections['foo']['fieldSets'][0]['items']);
    }

    public function testItShouldNotAddCheckboxesIfAllItemsAreAlsoInQueryOverride(): void
    {
        $this->inject($this->subject, 'filterSections', [
            'foo' => [
                'fieldSets' => [
                    [
                        'type' => 'checkboxes',
                        'items' => [ 'foo', 'bar', 'baz' ]
                    ]
                ]
            ]
        ]);
        $filterSections = $this->subject->getFilterSections([], [ 'preselectedTopics' => [ 'foo', 'bar', 'baz' ] ]);
        $this->assertArrayNotHasKey('foo', $filterSections);
    }

    public function testItShouldNotAddCheckboxesIfVariantsIsSetEventIfNotAllItemsInQueryOverride(): void
    {
        $this->inject($this->subject, 'filterSections', [
            'foo' => [
                'fieldSets' => [
                    [
                        'type' => 'checkboxes',
                        'items' => [ 'foo', 'bar', 'baz' ],
                        'variants' => true
                    ]
                ]
            ]
        ]);
        $filterSections = $this->subject->getFilterSections([], [ 'preselectedTopics' => [ 'baz' ] ]);
        $this->assertArrayNotHasKey('foo', $filterSections);
    }

    public function testItShouldAlwaysAddRangeTypes(): void
    {
        $this->inject($this->subject, 'filterSections', [
            'foo' => [
                'fieldSets' => [
                    [ 'type' => 'range', ]
                ]
            ]
        ]);
        $filterSections = $this->subject->getFilterSections([], []);
        $this->assertArrayHasKey('foo', $filterSections);
    }

    public function testItShouldAlwaysAddUnknownTypes(): void
    {
        $this->inject($this->subject, 'filterSections', [
            'foo' => [
                'fieldSets' => [
                    [ 'type' => 'lalalalal unknown', ]
                ]
            ]
        ]);
        $filterSections = $this->subject->getFilterSections([], []);
        $this->assertArrayHasKey('foo', $filterSections);
    }
}
