<?php declare(strict_types=1);

namespace Newland\Toubiz\Events\Neos\Service;

use Neos\Flow\Annotations as Flow;

/**
 * @Flow\Scope("singleton")
 */
class FilterConfiguration
{

    /**
     * @var array
     * @Flow\InjectConfiguration("filter.sections")
     */
    protected $filterSections;

    public function getFilterSections(array $queryOverride, array $properties): array
    {
        $filterSections = [];

        foreach ($this->filterSections as $filterSectionKey => $filterSectionConfig) {
            $fieldSets = $this->prepareFieldSets(
                $filterSectionConfig['fieldSets'] ?? [],
                $queryOverride,
                $properties
            );
            if ($fieldSets !== null) {
                $filterSections[$filterSectionKey]['fieldSets'] = $fieldSets;
            }
        }

        return $filterSections;
    }

    private function prepareFieldSets(array $fieldSets, array $queryOverride, array $properties): ?array
    {
        $resultingFieldSets = [];
        foreach ($fieldSets as $fieldSetKey => $fieldSetConfig) {
            $configToAdd = $this->prepareSingleFieldSetConfiguration($fieldSetConfig, $queryOverride, $properties);
            if ($configToAdd !== null) {
                $resultingFieldSets[$fieldSetKey] = $configToAdd;
            }
        }
        return empty($resultingFieldSets) ? null : $resultingFieldSets;
    }

    private function prepareSingleFieldSetConfiguration(
        array $fieldSetConfig,
        array $queryOverride,
        array $properties
    ): ?array {
        $type = $fieldSetConfig['type'] ?? null;
        switch ($type) {
            case 'categories':
                return $this->prepareCategoryFieldSetConfiguration($fieldSetConfig, $queryOverride);
            case 'regions':
                return $this->prepareRegionsFieldSetConfiguration($fieldSetConfig, $queryOverride);
            case 'checkboxes':
                return $this->prepareCheckboxesFieldSetConfiguration($fieldSetConfig, $properties);
            case 'range':
            default:
                return $fieldSetConfig;
        }
    }

    private function prepareCategoryFieldSetConfiguration(array $fieldSetConfig, array $queryOverride): ?array
    {
        return empty($queryOverride['categories']) ? $fieldSetConfig : null;
    }

    private function prepareRegionsFieldSetConfiguration(array $fieldSetConfig, array $queryOverride): ?array
    {
        return empty($queryOverride['preselectedRegions']) ? $fieldSetConfig : null;
    }

    private function prepareCheckboxesFieldSetConfiguration(array $fieldSetConfig, array $properties): ?array
    {
        if (empty($fieldSetConfig['items'])) {
            return null;
        }

        // If an attribute is preselected, we have to check if the field set is a list of variants.
        // For variants, all other variant options are also hidden.
        // If the field set is not a set of variants, we only hide single options.
        if ($fieldSetConfig['variants'] ?? false) {
            $isItemPreselected = \count(
                array_intersect($fieldSetConfig['items'], $properties['preselectedTopics'])
            ) > 0;
            return $isItemPreselected ? null : $fieldSetConfig;
        }

        $fieldSetConfig['items'] = array_diff($fieldSetConfig['items'], $properties['preselectedTopics']);
        return empty($fieldSetConfig['items']) ? null : $fieldSetConfig;
    }
}
