<?php
namespace Newland\Toubiz\Events\Neos\ViewHelpers\Link;

use Neos\ContentRepository\Domain\Model\NodeInterface;
use Neos\Flow\Mvc\ActionRequest;
use Neos\FluidAdaptor\Core\ViewHelper\AbstractTagBasedViewHelper;
use Neos\Flow\Annotations as Flow;
use Neos\Neos\Service\LinkingService;
use Newland\NeosCommon\Service\NodeService;

/**
 * ViewHelper that generates links to frontend routes.
 * Frontend routes use the `Frontend\Node` controller of the `Neos.Neos` package
 * while passing the 'true' package & controller information as arguments.
 *
 * This allows for plugins to exist in the frontend context (with a page frame around
 * and on a specific node).
 * However, that also means that links have to be created in the following way, which
 * is very verbose:
 *
 * <neos:link.node node="{targetNode}"
 *      arguments="{
 *          --newland_toubiz_events_neos-full: {
 *              '@package': 'newland.toubiz.events.neos',
 *              '@controller': 'events',
 *              '@action': 'show',
 *              event: eventDate.event
 *          }
 *      }"
 *      format="html"
 *      class="c-button"
 *      additionalAttributes="{'itemprop': 'url'}">
 *
 *      ...
 * </neos:link.node>
 *
 * This ViewHelper aims to have a similar interface to `f:link.action` but for frontend routes.
 * The required argument `prefix` is the argument prefix under which the controller arguments
 * are being passed (in the example above that would be `--newland_toubiz_events_neos-full`).
 *
 * Similar to `f:link.action` this viewhelper automatically populates `controller` and
 * `package` if they are not passed.
 *
 * @example
 * <events:link.frontendRoute
 *      prefix="--newland_toubiz_events_neos-full"
 *      node="{targetNode}"
 *      action="show"
 *      arguments="{ event: eventDate.event }"
 *      class="c-button"
 *      additionalAttributes="{'itemprop': 'url'}">
 *
 *      ...
 * </events:link.frontendRoute>
 *
 * TODO Check if this view helper is even used anymore
 */
class FrontendRouteViewHelper extends AbstractTagBasedViewHelper
{
    protected $tagName = 'a';

    /**
     * @var LinkingService
     * @Flow\Inject
     */
    protected $linkingService;

    /**
     * @var NodeService
     * @Flow\Inject()
     */
    protected $nodeService;

    public function initializeArguments()
    {
        $this->registerUniversalTagAttributes();
    }

    public function render(
        NodeInterface $node,
        string $prefix,
        string $action,
        string $controller = null,
        string $package = null,
        array $arguments = [],
        string $format = 'html',
        bool $absolute = true
    ): string {
        $request = $this->controllerContext->getRequest();

        if ($request instanceof ActionRequest) {
            $controller = $controller ?: $request->getControllerName();
            $package = $package ?: $request->getControllerPackageKey();
        }

        $arguments['@action'] = $action;
        $arguments['@controller'] = $controller;
        $arguments['@package'] = $package;

        $uri = $this->linkingService->createNodeUri(
            $this->controllerContext,
            $this->nodeService->getDocumentNode($node),
            null,
            $format,
            $absolute,
            [ $prefix => $arguments ]
        );

        $this->tag->addAttribute('href', $uri);
        $this->tag->setContent($this->renderChildren());
        return $this->tag->render();
    }
}
