<?php declare(strict_types=1);
namespace Newland\Toubiz\Events\Neos\Search\Indexer;

/*
 * This file is part of the "toubiz-events-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Newland\Toubiz\Search\Neos\Indexer\AbstractRecordIndexer;
use Newland\Toubiz\Search\Neos\Indexer\IndexerInterface;
use Newland\Toubiz\Search\Neos\Indexer\IndexRecordModification;
use Newland\Toubiz\Sync\Neos\Domain\Model\Event;
use Newland\Toubiz\Sync\Neos\Domain\Filter\EventFilter;
use Newland\Toubiz\Sync\Neos\Domain\Repository\EventRepository;

/**
 * Event indexer.
 *
 * Indexes events for the search.
 *
 * @Flow\Scope("singleton")
 */
class EventIndexer extends AbstractRecordIndexer implements IndexerInterface
{
    public const SCOPE = 'toubiz_events';

    /**
     * @var EventRepository
     * @Flow\Inject
     */
    protected $eventRepository;

    /** @var int|null */
    private $total = null;

    /** @var int */
    private $processed = 0;

    public function index(): void
    {
        $this->eventRepository->withoutLanguageHandling(function () {
            $filter = $this->filter();
            $this->initializeCounters($filter);

            $this->eventRepository->forEachFilter($filter, function (Event $event) {
                $this->indexEvent($event);
                $this->informProgress(++$this->processed, $this->total);
            });
        });
    }

    public function postIndex(): void
    {
        $this->searchIndexRepository->deleteObsolete($this->index, static::SCOPE);

        // For legacy systems: Scope used to be classname.
        $this->searchIndexRepository->deleteObsolete($this->index, \get_class($this));
    }

    private function indexEvent(Event $event): void
    {
        $content = [
            $event->getTitle(),
            $event->getDescription(),
        ];

        $modification = new IndexRecordModification(
            (string) $event->getPersistenceObjectIdentifier(),
            $this->convertToIndexableString($event->getTitle()),
            $this->convertToIndexableString((string) $event->getDescription()),
            $this->convertToIndexableString(implode(' ', $content))
        );
        $modification->setLanguage($event->getLanguage())
            ->setScope(static::SCOPE);

        $this->addToIndex($modification);
    }

    private function filter(): EventFilter
    {
        $today = new \DateTime();
        $today->setTime(0, 0, 0);

        $filter = new EventFilter();
        $filter->setFromDate($today);
        return $filter;
    }

    private function initializeCounters(EventFilter $filter): void
    {
        $this->processed = 0;
        $this->total = $this->eventRepository->countByFilter($filter, 1)['items'] ?? null;
    }
}
