<?php
namespace Newland\Toubiz\Events\Neos\Domain\Repository;

use Doctrine\ORM\AbstractQuery;
use Newland\NeosCommon\Domain\Repository\TopicRepository as CommonTopicRepository;
use Newland\NeosCommon\Domain\Model\Topic;
use Newland\Toubiz\Api\Constants\Language;
use Newland\Toubiz\Sync\Neos\Domain\Filter\CategoryFilter;
use Newland\Toubiz\Sync\Neos\Domain\Filter\EventTagFilter;
use Newland\Toubiz\Sync\Neos\Domain\Model\Category;
use Newland\Toubiz\Sync\Neos\Domain\Model\EventTag;
use Newland\Toubiz\Sync\Neos\Domain\Repository\CategoryRepository;
use Neos\Flow\Annotations as Flow;
use Newland\Toubiz\Sync\Neos\Domain\Repository\EventTagRepository;

/**
 * @Flow\Scope("singleton")
 */
class TopicRepository extends CommonTopicRepository
{
    const TYPE_CATEGORY = 'category';
    const TYPE_TAG = 'eventTag';
    const TYPE_REGION = 'region';

    /**
     * Types of topics this repository handles.
     * For all types in this array there must be a `findAll{Type}` and `findBy{Type}`
     * method.
     *
     * @var string[]
     */
    protected $types = [ self::TYPE_CATEGORY, self::TYPE_TAG ];

    /**
     * @var CategoryRepository
     * @Flow\Inject()
     */
    protected $categoryRepository;


    /**
     * @var EventTagRepository
     * @Flow\Inject()
     */
    protected $eventTagRepository;

    public function setLanguage(string $language): void
    {
        Language::throwIfInvalid($language);

        // Topics themselves are transient.
        // Categories and tags however need to be filtered by language.
        $this->categoryRepository->setLanguage($language);
        $this->eventTagRepository->setLanguage($language);
    }

    public function findAllCategory()
    {
        $filter = new CategoryFilter();
        $filter->setOrderBy([ 'category.title' => 'ASC' ]);

        return array_map(
            function (Category $category) {
                return $this->topicFromCategory($category);
            },
            (array) $this->categoryRepository->findByFilter($filter)
        );
    }

    public function findAllCategorySortedByNumberOfEvents()
    {
        $queryBuilder = $this->categoryRepository->createQueryBuilder('category');
        $query = $queryBuilder
            ->select(
                [
                    'category.Persistence_Object_Identifier as identifier',
                    'category.title as title'
                ]
            )->groupBy('category.Persistence_Object_Identifier')
            ->innerJoin('category.events', 'event')
            ->orderBy('COUNT(event.Persistence_Object_Identifier)', 'DESC')
            ->getQuery()
            ->setHydrationMode(AbstractQuery::HYDRATE_ARRAY);

        return $this->cache('event_topics', function () use ($query) {
            return array_map(
                function (array $row) {
                    return new Topic(self::TYPE_CATEGORY, $row['identifier'], $row['title']);
                },
                $query->execute()
            );
        });
    }

    public function findByCategory(string $categoryId)
    {
        $category = $this->categoryRepository->findByIdentifier($categoryId);
        if (!$category) {
            return null;
        }

        return $this->topicFromCategory($category);
    }

    public function findAllEventTag()
    {
        $filter = new EventTagFilter();
        $filter->setOrderBy([ 'eventTag.keyword' => 'ASC' ]);

        return array_map(
            function (EventTag $eventTag) {
                return $this->topicFromTag($eventTag);
            },
            (array) $this->eventTagRepository->findByFilter($filter)
        );
    }

    public function findByEventTag(string $tagId)
    {
        $eventTag = $this->eventTagRepository->findByIdentifier($tagId);
        if (!$eventTag) {
            return null;
        }

        return $this->topicFromTag($eventTag);
    }

    private function topicFromCategory(Category $category): Topic
    {
        return new Topic(
            self::TYPE_CATEGORY,
            (string) $category->getPersistenceObjectIdentifier(),
            $category->getTitle()
        );
    }

    private function topicFromTag(EventTag $eventTag): Topic
    {
        return new Topic(
            self::TYPE_TAG,
            (string) $eventTag->getPersistenceObjectIdentifier(),
            $eventTag->getKeyword()
        );
    }
}
