<?php

/*
 * This file is part of the "toubiz-poi-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

namespace Newland\Toubiz\Events\Neos\ViewHelpers\Filter\Controller;

use Neos\ContentRepository\Domain\Model\Node;
use Neos\Flow\Annotations as Flow;
use Neos\FluidAdaptor\Core\Widget\AbstractWidgetController;
use Newland\NeosCommon\Domain\Model\Topic;
use Newland\NeosCommon\Service\NodeService;
use Newland\Toubiz\Events\Neos\Domain\Repository\TagRepository;
use Newland\Toubiz\Events\Neos\Service\RegionService;
use Newland\Toubiz\Sync\Neos\Domain\Filter\CategoryFilter;
use Newland\Toubiz\Sync\Neos\Domain\Model\Category;
use Newland\Toubiz\Sync\Neos\Domain\Model\EventTag;
use Newland\Toubiz\Sync\Neos\Domain\Repository\CategoryRepository;
use Newland\Toubiz\Sync\Neos\Domain\Repository\EventTagRepository;

class TagController extends AbstractWidgetController
{
    /**
     * @var CategoryRepository
     * @Flow\Inject
     */
    protected $categoryRepository;

    /**
     * @var CategoryFilter
     * @Flow\Inject
     */
    protected $categoryFilter;

    /**
     * @var TagRepository
     * @Flow\Inject
     */
    protected $tagRepository;

    /**
     * @var EventTagRepository
     * @Flow\Inject
     */
    protected $eventTagRepository;

    /**
     * @var RegionService
     * @Flow\Inject
     *
     */
    protected $regionService;

    /**
     * @var Node
     */
    protected $node;

    /**
     * @var array
     */
    private $tags = [];

    /**
     * @var array
     */
    private $parameters = [];

    /**
     * @var array
     */
    private $queryOverride = [];

    /**
     * @var NodeService
     * @Flow\Inject()
     */
    protected $nodeService;

    public function indexAction()
    {
        $this->initialize();

        if ($this->node !== null && !empty($this->parameters)) {
            $this->createTags();
        }

        $this->view->assignMultiple(
            [
                'tags' => $this->tags,
                'node' => $this->node,
                'documentNode' => $this->nodeService->getDocumentNode($this->node),
            ]
        );
    }

    private function initialize()
    {
        $this->node = $this->widgetConfiguration['node'] ?? null;
        $this->parameters = $this->widgetConfiguration['queryParameters'] ?? [];
        $this->queryOverride = $this->widgetConfiguration['queryOverride'] ?? [];
    }


    private function createTags()
    {
        $this->tags = [];
        $this->parseCategories();
        $this->parseTags();
        $this->parseRegions();
    }

    private function parseCategories()
    {
        foreach ($this->getCategories() as $category) {
            $this->tags[] = $this->tagRepository->findOneByCategory($category);
        }
    }

    /**
     * @return Category[]
     */
    private function getCategories(): array
    {
        $categories = [];
        if (array_key_exists('categories', $this->parameters) && $this->hasOverrideFor('categories') === false) {
            foreach ((array) $this->parameters['categories'] as $identifier) {
                $categories[] = $this->categoryRepository->findByIdentifier($identifier);
            }
        }

        return $categories;
    }

    private function parseTags()
    {
        foreach ($this->getTags() as $tag) {
            $this->tags[] = $this->tagRepository->findOneByEventTag($tag);
        }
    }

    /**
     * @return EventTag[]
     */
    private function getTags()
    {
        $tags = [];

        if (array_key_exists('eventTags', $this->parameters) && $this->hasOverrideFor('eventTags') === false) {
            foreach ((array) $this->parameters['eventTags'] as $identifier) {
                $tags[] = $this->eventTagRepository->findByIdentifier($identifier);
            }
        }

        return $tags;
    }

    private function parseRegions()
    {
        foreach ($this->getRegions() as $region) {
            $this->tags[] = $this->tagRepository->findOneByRegion($region);
        }
    }

    /**
     * @return Topic[]
     */
    private function getRegions(): array
    {
        $regions = [];

        if (array_key_exists('regions', $this->parameters) && $this->hasOverrideFor('preselectedRegions') === false) {
            foreach ((array) $this->parameters['regions'] as $identifier) {
                if ($this->regionService->findByIdentifier($identifier) !== null) {
                    $regions[] = $this->regionService->findByIdentifier($identifier);
                }
            }
        }

        return $regions;
    }

    /**
     * If some filters have an override, they are fixed and should not display tags - because tags
     * can change the query (remove the respective filter on click).
     *
     * @param string $filterSection
     * @return bool
     */
    private function hasOverrideFor(string $filterSection): bool
    {
        return !empty($this->queryOverride[$filterSection]);
    }
}
