<?php
namespace Newland\Toubiz\Events\Neos\Domain\Repository;

use Newland\NeosCommon\Domain\Repository\TopicRepository as CommonTopicRepository;
use Newland\NeosCommon\Domain\Model\Topic;
use Newland\Toubiz\Sync\Neos\Domain\Filter\CategoryFilter;
use Newland\Toubiz\Sync\Neos\Domain\Filter\EventTagFilter;
use Newland\Toubiz\Sync\Neos\Domain\Model\Category;
use Newland\Toubiz\Sync\Neos\Domain\Model\EventTag;
use Newland\Toubiz\Sync\Neos\Domain\Repository\CategoryRepository;
use Neos\Flow\Annotations as Flow;
use Newland\Toubiz\Sync\Neos\Domain\Repository\EventTagRepository;

/**
 * @Flow\Scope("singleton")
 */
class TopicRepository extends CommonTopicRepository
{
    const TYPE_CATEGORY = 'category';
    const TYPE_TAG = 'eventTag';

    /**
     * Types of topics this repository handles.
     * For all types in this array there must be a `findAll{Type}` and `findBy{Type}`
     * method.
     *
     * @var string[]
     */
    protected $types = [ self::TYPE_CATEGORY, self::TYPE_TAG ];

    /**
     * @var CategoryRepository
     * @Flow\Inject()
     */
    protected $categoryRepository;


    /**
     * @var EventTagRepository
     * @Flow\Inject()
     */
    protected $eventTagRepository;

    public function findAllCategory()
    {
        $filter = new CategoryFilter();
        $filter->setOrderBy([ 'category.title' => 'ASC' ]);

        return array_map(
            function (Category $category) {
                return $this->topicFromCategory($category);
            },
            $this->categoryRepository->findByFilter($filter)
        );
    }

    public function findByCategory(string $categoryId)
    {
        $category = $this->categoryRepository->findByIdentifier($categoryId);
        if (!$category) {
            return null;
        }

        return $this->topicFromCategory($category);
    }

    public function findAllEventTag()
    {
        $filter = new EventTagFilter();
        $filter->setOrderBy([ 'eventTag.keyword' => 'ASC' ]);

        return array_map(
            function (EventTag $eventTag) {
                return $this->topicFromTag($eventTag);
            },
            $this->eventTagRepository->findByFilter($filter)
        );
    }

    public function findByEventTag(string $tagId)
    {
        $eventTag = $this->eventTagRepository->findByIdentifier($tagId);
        if (!$eventTag) {
            return null;
        }

        return $this->topicFromTag($eventTag);
    }

    private function topicFromCategory(Category $category): Topic
    {
        return new Topic(
            self::TYPE_CATEGORY,
            $category->getPersistenceObjectIdentifier(),
            $category->getTitle()
        );
    }

    private function topicFromTag(EventTag $eventTag): Topic
    {
        return new Topic(
            self::TYPE_TAG,
            $eventTag->getPersistenceObjectIdentifier(),
            $eventTag->getKeyword()
        );
    }
}
