<?php
namespace Newland\Toubiz\Events\Neos\Routing;

/*
 * This file is part of the "toubiz-events-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Neos\Flow\Mvc\Routing\AbstractRoutePart;
use Neos\Flow\Mvc\Routing\Dto\MatchResult;
use Neos\Flow\Mvc\Routing\Dto\ResolveResult;
use Neos\Flow\Mvc\Routing\DynamicRoutePartInterface;
use Newland\PageFrameProvider\Definition\RoutingConfigurationHelper;
use Newland\PageFrameProvider\Service\PageFrameLinkingService;

/**
 * Event Type Routing handler.
 *
 * Handles URLs that are built for EventsController::showAction inside of a PageFrame.
 * The URLs can be configured in the `Newland.Toubiz.Events.Neos.detailPage` path which can contain
 * the event type as a key and the URL as a value.
 *
 * This Handler is tightly coupled to `ArticleUrlService` which generates URLs handled by this Route
 * Handler.
 *
 * @Flow\Scope("prototype")
 */
class EventDetailRoutePart extends AbstractRoutePart implements DynamicRoutePartInterface
{
    // Constants that identify the events plugin.
    const PACKAGE = 'newland.toubiz.events.neos';
    const CONTROLLER = 'events';
    const ACTION = 'show';

    // Plugin namespace as setup by the Fusion Object.
    const ARGUMENT_NAMESPACE = PageFrameLinkingService::DEFAULT_ARGUMENT_NAMESPACE;

    // The plugin argument that is used to determine the article type.
    const TYPE_ARGUMENT = '_type';

    // Configuration key to use if there is no detail URL for the specific article type.
    const TYPE_FALLBACK = '_fallback';

    // Url that is used if there is not even a `_fallback` configured (instead of simply failing altogether).
    const ULTIMATE_FALLBACK_URL = 'event/detail';

    /**
     * @var array
     * @Flow\InjectConfiguration(package="Newland.Toubiz.Events.Neos", path="detailPage")
     */
    protected $types;

    /**
     * @var PageFrameLinkingService
     * @Flow\Inject()
     */
    protected $pageFrameLinkingService;

    /**
     * @var RoutingConfigurationHelper
     * @Flow\Inject()
     */
    protected $configurationHelper;

    /**
     * Checks whether this Route Part corresponds to the given $routePath.
     * This method does not only check if the Route Part matches. It can also
     * shorten the $routePath by the matching substring when matching is successful.
     * This is why $routePath has to be passed by reference.
     *
     * @param string $routePath The request path to be matched - without query parameters, host and fragment.
     * @return bool|MatchResult true or an instance of MatchResult if Route Part matched $routePath, otherwise false.
     */
    public function match(&$routePath)
    {
        $this->configurationHelper->setTypeConfiguration($this->types);

        $routePathBefore = $routePath;
        $routePath = $this->configurationHelper->removeUriSegmentsFromRoutePath($routePath);

        return $routePathBefore !== $routePath;
    }

    /**
     * Checks whether this Route Part corresponds to the given $routeValues.
     * This method does not only check if the Route Part matches. It also
     * removes resolved elements from $routeValues-Array.
     * This is why $routeValues has to be passed by reference.
     *
     * @param array $routeValues An array with key/value pairs to be resolved by Dynamic Route Parts.
     * @return bool|ResolveResult true or an instance of ResolveResult if Route Part can resolve one or more
     *     $routeValues elements, otherwise false.
     */
    public function resolve(array &$routeValues)
    {
        $this->configurationHelper->setTypeConfiguration($this->types);
        if (!$this->isTryingToBuildEventUrl($routeValues)) {
            return false;
        }

        $type = $routeValues[static::ARGUMENT_NAMESPACE][static::TYPE_ARGUMENT] ?? static::TYPE_FALLBACK;
        unset($routeValues[static::ARGUMENT_NAMESPACE][static::TYPE_ARGUMENT]);
        return new ResolveResult(
            $this->configurationHelper->urlFragmentForType($type) ?? static::ULTIMATE_FALLBACK_URL
        );
    }

    public function setSplitString($splitString)
    {
        // Does nothing.
    }

    /**
     * @param array $routeValues
     * @return bool
     */
    private function isTryingToBuildEventUrl(array $routeValues): bool
    {
        $type = $routeValues[static::ARGUMENT_NAMESPACE][static::TYPE_ARGUMENT] ?? null;

        $isPageFrameActionLink = $this->pageFrameLinkingService->isPageFrameActionLink(
            $routeValues,
            static::PACKAGE,
            static::CONTROLLER,
            static::ACTION,
            static::ARGUMENT_NAMESPACE
        );

        return $isPageFrameActionLink && $type !== null;
    }
}
