<?php declare(strict_types=1);

namespace Newland\Toubiz\Events\Neos\Tests\Unit\Filter\Items;

use Doctrine\ORM\Query;
use Neos\Flow\Configuration\ConfigurationManager;
use Neos\Flow\Tests\FunctionalTestCase;
use Newland\NeosTestingHelpers\InteractsWithNodes;
use Newland\Toubiz\Events\Neos\Filter\EventDateFilterFactory;
use Newland\Toubiz\Events\Neos\Filter\Items\EventDateRoot;
use Newland\Toubiz\Sync\Neos\Domain\Model\AbstractEntity;
use Newland\Toubiz\Sync\Neos\Domain\Model\Event;
use Newland\Toubiz\Sync\Neos\Tests\Factory\EventDateFactory;
use Newland\Toubiz\Sync\Neos\Tests\Factory\EventFactory;

class EventHighlightTest extends FunctionalTestCase
{
    use InteractsWithNodes;

    protected static $testablePersistenceEnabled = true;

    public function testOnlyFindsEventDatesBelongingToHighlightEvents(): void
    {
        $highlightEvent = (new EventFactory($this->objectManager))->create([ 'isHighlight' => true ]);
        $nonHighlightEvent = (new EventFactory($this->objectManager))->create([ 'isHighlight' => false ]);
        $highlightDate = (new EventDateFactory($this->objectManager))->create(
            [ 'event' => $highlightEvent, 'beginsAt' => '2050-01-01T14:00' ]
        );
        $nonHighlightDate = (new EventDateFactory($this->objectManager))->create(
            [ 'event' => $nonHighlightEvent, 'beginsAt' => '2050-01-01T14:00' ]
        );

        $filter = $this->filter();
        $filter->setState([ 'isHighlight' => true ]);
        $ids = $this->ids($filter);

        $this->assertContains($highlightDate->getPersistenceObjectIdentifier(), $ids);
        $this->assertNotContains($nonHighlightDate->getPersistenceObjectIdentifier(), $ids);
    }

    public function testFindsAllEventsIfNotActive(): void
    {
        $highlightEvent = (new EventFactory($this->objectManager))->create([ 'isHighlight' => true ]);
        $nonHighlightEvent = (new EventFactory($this->objectManager))->create([ 'isHighlight' => false ]);
        $highlightDate = (new EventDateFactory($this->objectManager))->create(
            [ 'event' => $highlightEvent, 'beginsAt' => '2050-01-01T14:00' ]
        );
        $nonHighlightDate = (new EventDateFactory($this->objectManager))->create(
            [ 'event' => $nonHighlightEvent, 'beginsAt' => '2050-01-01T14:00' ]
        );

        $filter = $this->filter();
        $filter->setState([ 'isHighlight' => false ]);
        $ids = $this->ids($filter);

        $this->assertContains($highlightDate->getPersistenceObjectIdentifier(), $ids);
        $this->assertContains($nonHighlightDate->getPersistenceObjectIdentifier(), $ids);
    }

    /** @see TBNI-1050 */
    public function testDoesNotFindMultipleDatesOfSameHighlightEvent(): void
    {
        $highlightEvent = (new EventFactory($this->objectManager))->create([ 'isHighlight' => true ]);

        $highlightDates = [];
        $highlightDates[] = (new EventDateFactory($this->objectManager))->create(
            [ 'event' => $highlightEvent, 'beginsAt' => '2050-01-01T14:00' ]
        );
        $highlightDates[] = (new EventDateFactory($this->objectManager))->create(
            [ 'event' => $highlightEvent, 'beginsAt' => '2050-01-02T14:00' ]
        );
        $highlightDates[] = (new EventDateFactory($this->objectManager))->create(
            [ 'event' => $highlightEvent, 'beginsAt' => '2050-01-03T14:00' ]
        );
        $highlightDates[] = (new EventDateFactory($this->objectManager))->create(
            [ 'event' => $highlightEvent, 'beginsAt' => '2050-01-04T14:00' ]
        );

        $filter = $this->filter();
        $filter->setState([ 'isHighlight' => true ]);
        $ids = $this->ids($filter);

        // Only first should be found
        $this->assertContains($highlightDates[0]->getPersistenceObjectIdentifier(), $ids);
        $this->assertNotContains($highlightDates[1]->getPersistenceObjectIdentifier(), $ids);
        $this->assertNotContains($highlightDates[2]->getPersistenceObjectIdentifier(), $ids);
        $this->assertNotContains($highlightDates[3]->getPersistenceObjectIdentifier(), $ids);
    }

    /** @see TBNI-1050 */
    public function testDoesNotGroupIfNotActive(): void
    {
        $highlightEvent = (new EventFactory($this->objectManager))->create([ 'isHighlight' => true ]);

        $highlightDates = [];
        $highlightDates[] = (new EventDateFactory($this->objectManager))->create(
            [ 'event' => $highlightEvent, 'beginsAt' => '2050-01-01T14:00' ]
        );
        $highlightDates[] = (new EventDateFactory($this->objectManager))->create(
            [ 'event' => $highlightEvent, 'beginsAt' => '2050-01-02T14:00' ]
        );
        $highlightDates[] = (new EventDateFactory($this->objectManager))->create(
            [ 'event' => $highlightEvent, 'beginsAt' => '2050-01-03T14:00' ]
        );
        $highlightDates[] = (new EventDateFactory($this->objectManager))->create(
            [ 'event' => $highlightEvent, 'beginsAt' => '2050-01-04T14:00' ]
        );

        $filter = $this->filter();
        $filter->setState([ 'isHighlight' => false ]);
        $ids = $this->ids($filter);

        $this->assertContains($highlightDates[0]->getPersistenceObjectIdentifier(), $ids);
        $this->assertContains($highlightDates[1]->getPersistenceObjectIdentifier(), $ids);
        $this->assertContains($highlightDates[2]->getPersistenceObjectIdentifier(), $ids);
        $this->assertContains($highlightDates[3]->getPersistenceObjectIdentifier(), $ids);
    }

    private function filter(): EventDateRoot
    {
        $settings = $this->objectManager->get(ConfigurationManager::class)->getConfiguration(
            ConfigurationManager::CONFIGURATION_TYPE_SETTINGS,
            'Newland.Toubiz.Events.Neos.filter'
        );
        return (new EventDateFilterFactory($this->initializeNode('/sites/foo')))
            ->createEventFilter();
    }

    private function ids(EventDateRoot $filter): array
    {
        $result = $filter->getEventQuery()
            ->select('entity.Persistence_Object_Identifier')
            ->getQuery()
            ->execute([], Query::HYDRATE_ARRAY);
        return array_column($result, 'Persistence_Object_Identifier');
    }
}
