<?php declare(strict_types=1);

namespace Newland\Toubiz\Events\Neos;

use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\QueryBuilder;
use Neos\ContentRepository\Domain\Model\NodeInterface;
use Neos\Flow\Log\ThrowableStorageInterface;
use Neos\Flow\ObjectManagement\ObjectManager;
use Neos\Flow\Persistence\PersistenceManagerInterface;
use Neos\Fusion\Core\Cache\ContentCache;
use Newland\Contracts\Neos\Filter\FilterRoot;
use Newland\Toubiz\Events\Neos\Filter\EventDateFilterFactory;
use Newland\Toubiz\Events\Neos\Filter\Items\EventDateRoot;
use Newland\Toubiz\Sync\Neos\Command\Helper\ConfigurationHelper;
use Newland\Toubiz\Sync\Neos\Command\Task\SyncEventsFromToubizApiV1;
use Newland\Toubiz\Sync\Neos\Configuration\ConfigurationAction;
use Newland\Toubiz\Sync\Neos\Configuration\ConfigurationProvider as ConfigurationProviderInterface;
use Newland\Toubiz\Sync\Neos\Domain\Model\Event;
use Newland\Toubiz\Sync\Neos\Domain\Model\EventDate;
use Neos\Flow\Annotations as Flow;
use Newland\Toubiz\Sync\Neos\Domain\Repository\EventRepository;
use Newland\Toubiz\Sync\Neos\Logging\LoggerFactory;
use Psr\Log\LoggerInterface;

class ConfigurationProvider implements ConfigurationProviderInterface
{

    /** @var string */
    private $type;

    /**
     * @var array
     * @Flow\InjectConfiguration(path="filter")
     */
    protected $filterSettings;

    /**
     * @var EventRepository
     * @Flow\Inject()
     */
    protected $eventRepository;

    /**
     * @var ContentCache
     * @Flow\Inject()
     */
    protected $contentCache;

    /**
     * @var ConfigurationHelper
     * @Flow\Inject()
     */
    protected $configurationHelper;

    /**
     * @var PersistenceManagerInterface
     * @Flow\Inject()
     */
    protected $persistenceManager;

    /**
     * @var EntityManagerInterface
     * @Flow\Inject()
     */
    protected $entityManager;

    /**
     * @var ObjectManager
     * @Flow\Inject()
     */
    protected $objectManager;

    /**
     * @var LoggerInterface
     */
    protected $logger;

    /**
     * @var LoggerFactory
     */
    protected $loggerFactory;

    /**
     * @var ThrowableStorageInterface
     * @Flow\Inject()
     */
    protected $exceptionLogger;

    public function injectLogger(LoggerFactory $loggerFactory): void
    {
        $this->loggerFactory = $loggerFactory;
        $this->logger = $loggerFactory->getLogger();
    }

    /** @return string[] */
    public function types(): array
    {
        return [ 'event' ];
    }

    public function setType(string $type): void
    {
        $this->type = $type;
    }

    public function filter(NodeInterface $node): FilterRoot
    {
        return (new EventDateFilterFactory($node))->createEventFilter($this->filterSettings, [], 'backend');
    }

    /** @param EventDateRoot $filter */
    public function count(FilterRoot $filter): int
    {
        return (int) $this->eventQuery($filter, false)
            ->select('COUNT(wrappingEvent.Persistence_Object_Identifier) AS count')
            ->getQuery()
            ->execute()[0]['count'];
    }

    /** @param EventDateRoot $filter */
    public function fetch(FilterRoot $filter): array
    {
        return $this->eventQuery($filter, true)->getQuery()->execute();
    }

    protected function getFormattedDateRange(Event $event): string
    {
        if ($event->getBeginsAt() === null || $event->getEndsAt() === null) {
            return '';
        }

        return sprintf(
            '%s - %s',
            $event->getBeginsAt()->format('Y-m-d H:i'),
            $event->getEndsAt()->format('Y-m-d H:i')
        );
    }

    private function eventQuery(EventDateRoot $filter, bool $withPagination): QueryBuilder
    {
        $subQuery = $filter->getEventQuery([], true)
            ->select('entity.Persistence_Object_Identifier');

        $query = $this->eventRepository->createQueryBuilder('wrappingEvent', null, true);
        $query->join('wrappingEvent.eventDates', 'eventDate');
        $query->where(
            $query->expr()->in(
                'eventDate.Persistence_Object_Identifier',
                $subQuery->getDQL()
            )
        );
        $query->setParameters($subQuery->getParameters());
        $query->distinct(true);
        $query->orderBy('wrappingEvent.beginsAt', 'ASC');

        $pagination = $filter->getPagination();
        if ($withPagination && $pagination) {
            $pagination->modifyDatabaseQuery($query);
        }

        return $query;
    }

    /**
     * Converts the given fetched item to table items: An array of primitive values displayed in the table.
     *
     * @param Event $item
     */
    public function tableItems($item): array
    {
        $dates = $item->getEventDates()->map(
            function (EventDate $date) {
                if ($date->getBeginsAt() === null || $date->getEndsAt() === null) {
                    return '';
                }

                return sprintf(
                    '%s - %s',
                    $date->getBeginsAt()->format('Y-m-d H:i'),
                    $date->getEndsAt()->format('Y-m-d H:i')
                );
            }
        )->toArray();

        $titleAttribute = implode("\n", $dates);

        $datesHtml = sprintf(
            '<span title="%s">%s</span>',
            $titleAttribute,
            $this->getFormattedDateRange($item)
        );

        return [
            sprintf(
                '<small>%s</small> '
                . '<span data-copy-to-clipboard data-copy-to-clipboard-content="%s">'
                . '<i class="fas fa-clipboard" style="cursor: pointer"></i>'
                . '</span>',
                $item->getOriginalId(),
                $item->getOriginalId()
            ),
            $item->getUrlIdentifier(),
            $item->getTitle(),
            $datesHtml,
            $item->getUpdatedAt()->format('d.m.Y H:i'),
        ];
    }

    /**
     * @param Event $item
     * @return ConfigurationAction[]
     */
    public function actions($item): array
    {
        return [
            'hide' => new ConfigurationAction(
                $item->isHidden() ? 'anzeigen' : 'verbergen',
                $item->isHidden() ? 'fas fa-eye-slash' : 'fas fa-eye',
                $item->isHidden() ? 'red' : 'green',
                function (array $configuration) {
                    $configuration['hidden'] = !($configuration['hidden'] ?? false);
                    $this->contentCache->flushByTag('event');
                    return $configuration;
                }
            ),
            'sync' => new ConfigurationAction(
                'synchronisieren',
                'fas fa-retweet',
                'green',
                function (array $configuration) use ($item) {
                    $this->contentCache->flushByTag('event');
                    $task = new SyncEventsFromToubizApiV1();
                    $id = $item->getOriginalId();
                    $this->configurationHelper->setDefaults([
                        'concurrency' => 2,
                    ]);

                    $errorHandlerWrapper = function ($block) {
                        return function () use ($block) {
                            try {
                                $block(...\func_get_args());
                                $this->persistenceManager->persistAll();
                            } catch (\Throwable $e) {
                                $this->logger->critical($this->exceptionLogger->logThrowable($e));
                                if (!$this->entityManager->isOpen()
                                    || $this->objectManager->getContext()->isTesting()
                                ) {
                                    throw $e;
                                }
                            }
                        };
                    };

                    foreach ($task->configurations($this->configurationHelper) as $syncConfiguration) {
                        $task->synchronizeSingle($id, $syncConfiguration, $errorHandlerWrapper);
                    }

                    return $configuration;
                }
            ),
        ];
    }
}
