<?php
namespace Newland\Toubiz\Events\Neos\Service;

/*
 * This file is part of the "toubiz-events-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\ContentRepository\Domain\Model\Node;
use Neos\ContentRepository\Domain\Model\NodeInterface;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Mvc\Controller\ControllerContext;
use Newland\NeosCommon\Service\ControllerContextFactory;
use Newland\PageFrameProvider\Service\PageFrameLinkingService;
use Newland\Toubiz\Events\Neos\Routing\EventDetailRoutePart;
use Newland\Toubiz\Sync\Neos\Domain\Model\Event;

/**
 * Generates detail URLs for events.
 *
 * This Service is tightly coupled to the `PageFrameController` functionality of `Newland.PageFrameProvider` to have
 * detail pages that are independent of the PageTree.
 *
 * @Flow\Scope("singleton")
 */
class EventUrlService
{
    /**
     * @var ControllerContextFactory
     * @Flow\Inject
     */
    protected $controllerContextFactory;

    /**
     * @var PageFrameLinkingService
     * @Flow\Inject()
     */
    protected $pageFrameLinkingService;

    /**
     * @param Event $event
     * @param ControllerContext $context
     * @param NodeInterface|null $referenceNode
     * @return string|null
     */
    public function generateUrl(Event $event, ControllerContext $context, NodeInterface $referenceNode = null)
    {
        return $this->generateDetailViewUrl($event, $context, $referenceNode);
    }

    /**
     * Some requests are within a controller context that has no "current node".
     * So we pass the current node manually and create a fake controller context around that.
     *
     * @param Event $event
     * @param Node $node
     * @return mixed|string|null
     */
    public function generateUrlByCurrentNode(Event $event, Node $node)
    {
        $context = $this->controllerContextFactory->initializeFakeControllerContext($node);

        return $this->generateUrl($event, $context, $node);
    }

    private function generateDetailViewUrl(
        Event $event,
        ControllerContext $context,
        NodeInterface $referenceNode = null
    ): string {
        return $this->pageFrameLinkingService->build(
            $context->getUriBuilder(),
            EventDetailRoutePart::PACKAGE,
            EventDetailRoutePart::CONTROLLER,
            EventDetailRoutePart::ACTION,
            'eventDetail',
            [
                'event' => $event,
                // events don't currently have types
                EventDetailRoutePart::TYPE_ARGUMENT => 'default',
            ],
            EventDetailRoutePart::ARGUMENT_NAMESPACE,
            $referenceNode
        );
    }
}
