<?php
namespace Newland\Toubiz\Events\Neos\Controller;

/*
 * This file is part of the "toubiz-events-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\ContentRepository\Domain\Model\Node;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Configuration\Exception\InvalidConfigurationException;

use Newland\Toubiz\Events\Neos\Domain\Repository\TopicRepository;
use Newland\Toubiz\Events\Neos\Service\RegionService;
use Newland\Toubiz\Events\Neos\Encoder\TopicsToQueryEncoder;
use Newland\Toubiz\Sync\Neos\Domain\Model\Event;
use Newland\Toubiz\Sync\Neos\Domain\Model\EventDate;
use Newland\Toubiz\Sync\Neos\Domain\Repository\EventDateRepository;
use Newland\Toubiz\Sync\Neos\Domain\Filter\EventDateFilter;
use Newland\Toubiz\Events\Neos\Utility\DateTimeUtility;

/**
 * Events controller.
 *
 * Responsible for handling event data.
 *
 * @Flow\Scope("singleton")
 */
class EventsController extends AbstractActionController
{
    /**
     * @var int Items per page for pagination.
     */
    const ITEMS_PER_PAGE = 10;

    /**
     * @Flow\Inject
     * @var EventDateRepository
     */
    protected $eventDateRepository;

    /**
     * @var RegionService
     * @Flow\Inject()
     */
    protected $regionService;

    /**
     * @Flow\Inject
     * @var TopicRepository
     */
    protected $topicRepository;

    /**
     * @var array
     * @Flow\InjectConfiguration("filter.sections")
     */
    protected $filterSections;

    /**
     * @var array
     */
    protected $queryOverride = [];

    /**
     * @var TopicsToQueryEncoder
     * @Flow\Inject
     */
    protected $topicsToQueryEncoder;

    /**
     * Teaser action.
     *
     * Lists a subset of upcoming events.
     *
     * @return void
     */
    public function teaserAction()
    {
        $today = DateTimeUtility::buildDateTimeObject();

        $eventFilter = new EventDateFilter;
        $eventFilter->setFromDate($today)
            ->setOrderBy([ 'eventDate.beginsAt' => 'asc' ])
            ->setLimit($this->properties['recordLimit']);

        if ($this->properties['filterHighlights']) {
            $eventFilter->setHighlight(true);
        }

        $this->view->assignMultiple(
            [
                'currentDate' => $today,
                'eventDates' => $this->eventDateRepository->findByFilter($eventFilter),
            ]
        );

        if (array_key_exists('targetNode', $this->properties)) {
            $targetNode = $this->linkingService->convertUriToObject(
                $this->properties['targetNode'],
                $this->node
            );
            $this->view->assign('targetNode', $targetNode);
        }
    }

    /**
     * Index action.
     *
     * Displays an overview of events based on integration type.
     *
     * @return void
     */
    public function indexAction()
    {
        $this->applyQueryOverride();
        $this->assignDatePickerProperties();
        $topics = $this->topicRepository->findByIdentifiers($this->properties['topics'] ?? []);

        $this->view->assignMultiple(
            [
                'highlightEventDates' => $this->highlightedEvents(),
                'topics' => $topics,
                'queryOverride' => $this->queryOverride,
                'filterSections' => $this->generateSelectableFilterSections(),
                'datePickerLoadPage' => $this->getDatePickerLoadPage(),
            ]
        );

        $integrationTypeAction = 'indexActionFor' . ucwords($this->integrationType) . 'IntegrationType';
        if (!method_exists($this, $integrationTypeAction)) {
            throw new InvalidConfigurationException(
                'No ' . $integrationTypeAction . ' method found in ' . static::class . ' to support'
                . ' integration type ' . $this->integrationType
            );
        }
        $this->{$integrationTypeAction}();
    }

    private function indexActionForPointIntegrationType()
    {
        $this->view->assignMultiple(
            [
                'eventsToday' => $this->eventsToday(),
                'nextEventDates' => $this->eventsComingUp(),
            ]
        );
    }

    private function indexActionForDestinationIntegrationType()
    {
        $today = DateTimeUtility::buildDateTimeObject();
        $nextFilter = $this->filterForProperties()->setFromDate($today);

        $this->view->assign(
            'nextEventDates',
            $this->eventDateRepository->findByFilter($nextFilter)
        );
    }

    /**
     * @param array $query
     * @return void
     */
    public function searchResultAction(array $query = [])
    {
        $this->applyQueryOverride();
        $this->assignDatePickerProperties($query);
        $this->assignPropertiesForFilteredLists($query);
    }

    /**
     * @param Node $node
     * @param array $query
     */
    public function searchResultAjaxAction(Node $node, array $query = [])
    {
        $this->setupCustomNodeMappingForAjaxRequest($node);
        $this->applyQueryOverride();
        $this->assignDatePickerProperties($query);
        $this->assignPropertiesForFilteredLists($query);
    }

    /**
     * @param Event $event
     * @return void
     */
    public function showAction(Event $event = null)
    {
        if (!$event) {
            return;
        }

        $this->view->assign('event', $event);
    }

    private function assignPropertiesForFilteredLists(array $query)
    {
        $demand = $this->computeDemandFromQuery($query);
        $filter = $this->filterForProperties($demand);

        $page = array_key_exists('page', $query) ? (int) $query['page'] : 1;
        $this->view->assignMultiple(
            [
                'filterSections' => $this->filterSections,
                'eventDates' => $this->eventDateRepository->findByFilter($filter),
                'pagination' => $this->paginationProperties($filter, $page),
                'queryOverride' => $this->queryOverride,
                'query' => $query,
            ]
        );
    }

    private function assignDatePickerProperties(array $query = [])
    {
        $today = (new \DateTime())->format('Y-m-d');
        $tomorrow = (new \DateTime())->modify('+1 day')->format('Y-m-d');
        $datePickerMinDate = (new \DateTime())->format('m/d/Y');
        $datePickerIsActive = array_key_exists('fromDate', $query)
            && array_key_exists('toDate', $query)
            && $query['fromDate'] !== $query['toDate'];

        $this->view->assignMultiple(
            [
                'today' => $today,
                'tomorrow' => $tomorrow,
                'datePickerMinDate' => $datePickerMinDate,
                'datePickerIsActive' => $datePickerIsActive,
                'initialDatepickerState' => json_encode(
                    [
                        'start' => $datePickerIsActive ? ($query['fromDate'] ?? null) : null,
                        'end' => $datePickerIsActive ? ($query['toDate'] ?? null) : null,
                    ]
                ),
            ]
        );
    }

    private function filterForProperties(array $properties = null): EventDateFilter
    {
        $properties = $properties ?? $this->properties;
        $properties = $this->overrideQuery($properties, $this->queryOverride);
        $properties['page'] = array_key_exists('page', $properties) ? (int) $properties['page'] : 1;
        $properties['page'] = $properties['page'] > 1 ? $properties['page'] : 1;
        if (array_key_exists('categories', $properties)) {
            $properties['categories'] = array_values($properties['categories']);
        }
        if (array_key_exists('eventTags', $properties)) {
            $properties['eventTags'] = array_values($properties['eventTags']);
        }
        /** @var EventDateFilter $filter */
        $filter = (new EventDateFilter())
            ->initialize($properties)
            ->setOrderBy([ 'eventDate.beginsAt' => 'asc' ])
            ->setOffset(($properties['page'] - 1) * self::ITEMS_PER_PAGE)
            ->setLimit(self::ITEMS_PER_PAGE);

        if (array_key_exists('preselectedRegions', $properties)) {
            $filter->setLocationZips(
                $this->regionService->collectZipsFromRegionKeys($properties['preselectedRegions'], true)
            );
        }


        return $filter;
    }

    private function paginationProperties(EventDateFilter $filter, int $page): array
    {
        $eventsCount = $this->eventDateRepository->countByFilter($filter, self::ITEMS_PER_PAGE);
        return [
            'isFirst' => $page === 1,
            'page' => $page,
            'isLast' => $eventsCount['pages'] <= $page,
            'count' => $eventsCount,
        ];
    }

    private function computeDemandFromQuery(array $query, array $overrides = null)
    {
        $overrides = $overrides ?? $this->queryOverride;
        $demand = array_replace_recursive($overrides, $query);

        // Treat empty strings as non-existent
        return array_filter(
            $demand,
            function ($value) {
                return $value !== '';
            }
        );
    }

    /**
     * @return EventDate[]
     */
    private function highlightedEvents(): array
    {
        $today = DateTimeUtility::buildDateTimeObject();
        $demand = $this->computeDemandFromQuery([ 'highlight' => true, 'fromDate' => $today ]);
        $filter = $this->filterForProperties($demand)
            ->setOrderBy([ 'event.beginsAt' => 'asc' ])
            ->setGroupBy([ 'event.Persistence_Object_Identifier' ])
            ->setLimit(9);

        return $this->eventDateRepository->findByFilter($filter);
    }

    /**
     * @return EventDate[]
     */
    private function eventsToday(): array
    {
        $today = DateTimeUtility::buildDateTimeObject();
        $demand = $this->computeDemandFromQuery([ 'fromDate' => $today, 'toDate' => $today ]);
        $filter = $this->filterForProperties($demand)
            ->setOrderBy([ 'eventDate.beginsAt' => 'asc' ]);

        return $this->eventDateRepository->findByFilter($filter);
    }

    private function eventsComingUp(): array
    {
        $start = DateTimeUtility::buildDateTimeObject();
        $start->modify('+1 day');
        $end = DateTimeUtility::buildDateTimeObject();
        $end->modify('+7 days');

        $demand = $this->computeDemandFromQuery([ 'fromDate' => $start, 'fromMaxDate' => $end ]);
        $filter = $this->filterForProperties($demand)
            ->setOrderBy([ 'eventDate.beginsAt' => 'asc' ]);

        return $this->eventDateRepository->findByFilter($filter);
    }

    private function getDatePickerLoadPage(): string
    {
        return $this->getControllerContext()->getUriBuilder()
            ->reset()
            ->uriFor(
                'searchResult',
                [
                    'search' => [
                        'fromDate' => '{{=it.start}}',
                        'toDate' => '{{=it.end}}',
                    ],
                ]
            );
    }

    private function applyQueryOverride()
    {
        $preselectedTopics = (array) ($this->properties['preselectedTopics'] ?? []);
        $this->queryOverride = $this->topicsToQueryEncoder->encode($preselectedTopics);

        $regions = (array) ($this->properties['preselectedRegions'] ?? []);
        $this->queryOverride = array_merge_recursive($this->queryOverride, [ 'preselectedRegions' => $regions ]);
    }

    /**
     * If an override is set, then the relevant parts of the current query are ignored
     * and replaced with the override.
     *
     * @param array $query
     * @param array $override
     * @return array
     */
    private function overrideQuery(array $query = [], array $override = [])
    {
        foreach ($override as $key => $value) {
            $query[$key] = $value;
        }

        return $query;
    }

    private function generateSelectableFilterSections(): array
    {
        $filterSections = [];

        foreach ($this->filterSections as $filterSectionKey => $filterSectionConfig) {
            if (!is_array($filterSectionConfig) || !array_key_exists('fieldSets', $filterSectionConfig)) {
                continue;
            }

            $filterSection = [ 'fieldSets' => [] ];

            foreach ($filterSectionConfig['fieldSets'] as $fieldSetKey => $fieldSetConfig) {
                if ($fieldSetConfig['type'] === 'categories') {
                    // If a category is preselected, then we hide the field set.
                    if (!array_key_exists('categories', $this->queryOverride)) {
                        $filterSection['fieldSets'][$fieldSetKey] = $fieldSetConfig;
                    }
                } elseif ($fieldSetConfig['type'] === 'checkboxes'
                    && isset($fieldSetConfig['items'])
                    && is_array($fieldSetConfig['items'])
                ) {
                    // If an attribute is preselected, we have to check if the field set is a list of variants.
                    // For variants, all other variant options are also hidden.
                    // If the field set is not a set of variants, we only hide single options.
                    if (empty($fieldSetConfig['variants'])) {
                        // No variants here. Remove any preselected options and add rest of the field set.
                        $fieldSetConfig['items'] = array_diff(
                            $fieldSetConfig['items'],
                            $this->properties['preselectedTopics']
                        );
                        if (\count($fieldSetConfig['items'])) {
                            $filterSection['fieldSets'][$fieldSetKey] = $fieldSetConfig;
                        }
                    } elseif (isset($fieldSetConfig['variants'])
                        && $fieldSetConfig['variants']
                        && \count(array_intersect($fieldSetConfig['items'], $this->properties['preselectedTopics']))
                    ) {
                        // If one or more variant is preselected, we skip the entire fieldset.
                        continue;
                    } else {
                        // Nothing is preselected. Add the entire fieldset.
                        $filterSection['fieldSets'][$fieldSetKey] = $fieldSetConfig;
                    }
                } else {
                    // FieldSets of type "range" cannot be pre-filtered right now. So we always show the field set.
                    $filterSection['fieldSets'][$fieldSetKey] = $fieldSetConfig;
                }
            }

            if (!empty($filterSection['fieldSets'])) {
                $filterSections[$filterSectionKey] = $filterSection;
            }
        }

        return $filterSections;
    }
}
